import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "https://esm.sh/@supabase/supabase-js@2";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers": "authorization, x-client-info, apikey, content-type",
};

serve(async (req) => {
  if (req.method === "OPTIONS") {
    return new Response("ok", { headers: corsHeaders });
  }

  try {
    const { email, role, organization_id, organization_name } = await req.json();

    if (!email || !organization_id) {
      return new Response(
        JSON.stringify({ error: "Email and organization ID are required" }),
        {
          status: 400,
          headers: { ...corsHeaders, "Content-Type": "application/json" },
        }
      );
    }

    const supabaseClient = createClient(
      Deno.env.get("SUPABASE_URL") ?? "",
      Deno.env.get("SUPABASE_SERVICE_ROLE_KEY") ?? ""
    );

    // Create invite record in database
    const { data: invite, error: inviteError } = await supabaseClient
      .from("organization_invites")
      .insert({
        email,
        role,
        organization_id,
        status: "pending",
        expires_at: new Date(Date.now() + 7 * 24 * 60 * 60 * 1000), // 7 days from now
      })
      .select()
      .single();

    if (inviteError) throw inviteError;

    // Generate invite link
    const inviteLink = `${Deno.env.get("SITE_URL")}/accept-invite?token=${invite.id}`;

    // Send email using Resend API (make sure RESEND_API_KEY is set in your Supabase project)
    const resendApiKey = Deno.env.get("RESEND_API_KEY");
    if (!resendApiKey) {
      console.warn("RESEND_API_KEY is missing, email will not be sent.");
    } else {
      const emailResponse = await fetch("https://api.resend.com/emails", {
        method: "POST",
        headers: {
          Authorization: `Bearer ${resendApiKey}`,
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          from: "no-reply@yourdomain.com",
          to: email,
          subject: `You've been invited to join ${organization_name}`,
          html: `
            <div style="font-family: Arial, sans-serif; color: #333;">
              <h2>Organization Invitation</h2>
              <p>You've been invited to join <strong>${organization_name}</strong> as a <strong>${role}</strong>.</p>
              <p>Click the button below to accept the invitation:</p>
              <a href="${inviteLink}" style="
                display:inline-block;
                padding:12px 24px;
                background:#3b82f6;
                color:white;
                text-decoration:none;
                border-radius:6px;
                margin:20px 0;
              ">Accept Invitation</a>
              <p>This invitation link will expire in 7 days.</p>
              <hr>
              <p style="font-size:12px; color:#666;">If you didn’t request this invitation, you can safely ignore this email.</p>
            </div>
          `,
        }),
      });

      if (!emailResponse.ok) {
        console.error("Failed to send email:", await emailResponse.text());
      }
    }

    return new Response(
      JSON.stringify({
        message: "Invite saved successfully",
        inviteId: invite.id,
        emailSent: !!resendApiKey,
      }),
      {
        status: 200,
        headers: { ...corsHeaders, "Content-Type": "application/json" },
      }
    );
  } catch (error) {
    console.error("Error:", error);
    return new Response(
      JSON.stringify({ error: error.message }),
      {
        status: 500,
        headers: { ...corsHeaders, "Content-Type": "application/json" },
      }
    );
  }
});
