// supabaseClient.ts
import { createClient } from '@supabase/supabase-js';

// Vite uses import.meta.env instead of process.env
const supabaseUrl = import.meta.env.VITE_SUPABASE_URL;
const supabaseAnonKey = import.meta.env.VITE_SUPABASE_ANON_KEY;

// Check if WebCrypto is available
const isWebCryptoSupported = () => {
  try {
    return typeof window !== 'undefined' && 
           window.crypto && 
           window.crypto.subtle &&
           typeof window.crypto.subtle.digest === 'function';
  } catch (e) {
    return false;
  }
};

console.log('WebCrypto supported:', isWebCryptoSupported());

// Provide better error messages for development
if (!supabaseUrl || !supabaseAnonKey) {
  const errorMessage = `Missing Supabase environment variables:
  - VITE_SUPABASE_URL: ${supabaseUrl ? '***' : 'MISSING'}
  - VITE_SUPABASE_ANON_KEY: ${supabaseAnonKey ? '***' : 'MISSING'}
  
  Please check your .env file and make sure:
  1. You have a .env file in the root directory
  2. It contains VITE_SUPABASE_URL and VITE_SUPABASE_ANON_KEY
  3. You've restarted the development server after adding them`;

  console.error(errorMessage);
  
  // Don't throw in development to allow the app to load
  if (import.meta.env.PROD) {
    throw new Error('Missing Supabase environment variables in production');
  }
}

export const supabase = createClient(
  supabaseUrl || 'https://default-fallback.supabase.co',
  supabaseAnonKey || 'default-fallback-anon-key',
  {
    auth: {
      autoRefreshToken: true,
      persistSession: true,
      detectSessionInUrl: true,
      storage: typeof window !== 'undefined' ? localStorage : undefined,
      flowType: isWebCryptoSupported() ? 'pkce' : 'implicit',
    },
    global: {
      headers: {
        'X-Client-Info': 'reflect-evolve-app',
      },
    },
  }
);