import React, { useState } from "react";
import { supabase } from "../supabaseClient";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Loader2 } from "lucide-react";
import { useNavigate } from "react-router-dom";
import { Card } from "@/components/ui/card";
import { FaUser, FaUsers, FaCrown } from "react-icons/fa";

const Auth: React.FC = () => {
  const [isSignUp, setIsSignUp] = useState(false);
  const [authType, setAuthType] = useState<"individual" | "organization" | null>(null);
  const [name, setName] = useState("");
  const [email, setEmail ] = useState("");
  const [password, setPassword] = useState("");
  const [confirmPassword, setConfirmPassword] = useState("");
  const [loading, setLoading] = useState(false);
  const [message, setMessage] = useState<string | null>(null);
  const [error, setError] = useState<string | null>(null);
  const [selectedPackage, setSelectedPackage] = useState<'Basic' | 'Pro'>('Basic');
 
  const navigate = useNavigate();
  
  

  const handleAuth = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    setError(null);
    setMessage(null);

    // In your Auth component
// Add to your Auth component

// Modify your signup success handler
const handleSignupSuccess = async (userData: any) => {
  try {
    // Process any pending invite
    const pendingInvite = sessionStorage.getItem('pending_invite');
    if (pendingInvite) {
      const inviteData = JSON.parse(pendingInvite);
      
      // Add user to organization
      const { error: memberError } = await supabase
        .from('organization_members')
        .insert({
          user_id: userData.user.id,
          organization_id: inviteData.organization_id,
          role: inviteData.role,
        });

      if (!memberError) {
        // Mark invite as accepted
        await supabase
          .from('organization_invites')
          .update({
            status: 'accepted',
            accepted_at: new Date().toISOString(),
            auth_user_id: userData.user.id
          })
          .eq('token', inviteData.token);

        alert(`🎉 Welcome! You've joined ${inviteData.org_name} as ${inviteData.role}`);
      }

      sessionStorage.removeItem('pending_invite');
    }

    // Redirect user
    const accountType = userData.user.user_metadata?.account_type;
    navigate(accountType === 'organization' ? "/OrganizationDashboard" : "/IndividualDashboard");

  } catch (error) {
    console.error('Post-signup processing error:', error);
    navigate("/dashboard");
  }
};

    try {
      if (isSignUp) {
        if (password !== confirmPassword) {
          setError("Passwords do not match.");
          setLoading(false);
          return;
        }

        if (password.length < 6) {
          setError("Password must be at least 6 characters long.");
          setLoading(false);
          return;
        }

        // Set user metadata with package type
        const userMetadata: any = {
          name: name.trim(),
          account_type: authType,
          package_type: selectedPackage
        };

        const { data: authData, error: authError } = await supabase.auth.signUp({
          email: email.toLowerCase().trim(),
          password,
          options: {
            data: userMetadata
          }
        });

        if (authError) {
          console.error('Auth error:', authError);
          if (authError.message.includes('already registered') || authError.message.includes('user already exists')) {
            setError("This email is already registered. Please sign in instead.");
          } else {
            setError(authError.message);
          }
          setLoading(false);
          return;
        }

        if (!authData.user) {
          setError("Failed to create user account. Please try again.");
          setLoading(false);
          return;
        }

        setMessage(
          selectedPackage === 'Pro' 
            ? "✅ Pro account created! Career Development feature is now active. Please check your email to confirm your account."
            : "✅ Signup successful! Please check your email to confirm your account."
        );
        
      } else {
        // Login flow
        const { data, error: loginError } = await supabase.auth.signInWithPassword({
          email: email.toLowerCase().trim(),
          password,
        });

        if (loginError) {
          console.error('Login error:', loginError);
          
          if (loginError.message.includes('Invalid login credentials')) {
            setError("Invalid email or password. Please try again.");
          } else if (loginError.message.includes('Email not confirmed')) {
            setError("Please confirm your email address before signing in.");
          } else {
            setError(loginError.message);
          }
          
          setLoading(false);
          return;
        }

        // Redirect based on account type
        if (data.user) {
          const accountType = data.user.user_metadata?.account_type;
          navigate(accountType === 'organization' 
            ? "/OrganizationDashboard" 
            : "/IndividualDashboard");
        }
      }
    } catch (err: any) {
      console.error('Unexpected error:', err);
      setError("An unexpected error occurred. Please try again.");
    } finally {
      setLoading(false);
    }
  };

  // Check if user has career development access (Pro package)
  const hasCareerDevelopmentAccess = (packageType: string | null | undefined): boolean => {
    return packageType === 'Pro';
  };

  // Render auth type selection if not chosen yet
  if (!authType) {
    return (
      <section className="min-h-screen flex items-center justify-center bg-gradient-to-br from-indigo-600 via-purple-600 to-pink-500 p-4">
        <div className="bg-white text-gray-800 rounded-2xl shadow-lg p-8 w-full max-w-md">
          <h1 className="text-3xl font-bold mb-6 text-center">
            Choose Account Type
          </h1>
          
          <div className="grid grid-cols-1 gap-4">
            <Card 
              className="p-6 cursor-pointer hover:bg-gray-50 transition-colors"
              onClick={() => setAuthType("individual")}
            >
              <div className="flex flex-col items-center text-center">
                <FaUser className="text-4xl mb-3 text-indigo-600" />
                <h3 className="text-xl font-semibold">Individual Account</h3>
                <p className="text-gray-600 mt-1">
                  Choose from Basic or Pro packages
                </p>
              </div>
            </Card>
            
            <Card 
              className="p-6 cursor-pointer hover:bg-gray-50 transition-colors"
              onClick={() => setAuthType("organization")}
            >
              <div className="flex flex-col items-center text-center">
                <FaUsers className="text-4xl mb-3 text-purple-600" />
                <h3 className="text-xl font-semibold">Organization Account</h3>
                <p className="text-gray-600 mt-1">
                  For teams and organizations
                </p>
              </div>
            </Card>
          </div>
        </div>
      </section>
    );
  }

  return (
    <section className="min-h-screen flex items-center justify-center bg-gradient-to-br from-indigo-600 via-purple-600 to-pink-500 p-4">
      <div className="bg-white text-gray-800 rounded-2xl shadow-lg p-8 w-full max-w-md">
        <button 
          onClick={() => {
            setAuthType(null);
            setSelectedPackage('Basic');
            setError(null);
            setMessage(null);
          }}
          className="text-sm text-indigo-600 hover:underline mb-4 flex items-center"
        >
          ← Back to account selection
        </button>
        
        <h1 className="text-3xl font-bold mb-2 text-center">
          {isSignUp ? "Create Account" : "Welcome Back"}
        </h1>
        
        {isSignUp && authType === 'individual' && (
          <div className="mb-6">
            <div className="flex space-x-2 mb-4">
              {(['Basic', 'Pro'] as const).map((pkg) => (
                <button
                  key={pkg}
                  type="button"
                  onClick={() => setSelectedPackage(pkg)}
                  className={`flex-1 py-2 px-4 rounded-lg border-2 text-sm font-medium transition-all ${
                    selectedPackage === pkg
                      ? pkg === 'Pro'
                        ? 'border-yellow-500 bg-yellow-50 text-yellow-700'
                        : 'border-indigo-500 bg-indigo-50 text-indigo-700'
                      : 'border-gray-300 bg-white text-gray-600 hover:border-gray-400'
                  }`}
                >
                  <div className="flex items-center justify-center gap-1">
                    {pkg === 'Pro' && <FaCrown className="text-yellow-500" />}
                    {pkg}
                  </div>
                </button>
              ))}
            </div>
            
            {selectedPackage === 'Pro' && (
              <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-3">
                <p className="text-sm text-yellow-800 text-center">
                  <strong>Pro Feature:</strong> Career Development tools included
                </p>
              </div>
            )}
          </div>
        )}

        <form onSubmit={handleAuth} className="space-y-4">
          {isSignUp && (
            <div>
              <label className="text-sm font-medium">
                {authType === "organization" ? "Organization Name" : "Full Name"}
              </label>
              <Input
                type="text"
                placeholder={authType === "organization" ? "Organization Name" : "Your Name"}
                value={name}
                onChange={(e) => setName(e.target.value)}
                required
                minLength={2}
              />
            </div>
          )}

          <div>
            <label className="text-sm font-medium">Email</label>
            <Input
              type="email"
              placeholder="you@example.com"
              value={email}
              onChange={(e) => setEmail(e.target.value)}
              required
            />
          </div>

          <div>
            <label className="text-sm font-medium">Password</label>
            <Input
              type="password"
              placeholder="••••••••"
              value={password}
              onChange={(e) => setPassword(e.target.value)}
              required
              minLength={6}
            />
            {isSignUp && (
              <p className="text-xs text-gray-500 mt-1">
                Must be at least 6 characters long
              </p>
            )}
          </div>

          {isSignUp && (
            <div>
              <label className="text-sm font-medium">Confirm Password</label>
              <Input
                type="password"
                placeholder="Re-enter password"
                value={confirmPassword}
                onChange={(e) => setConfirmPassword(e.target.value)}
                required
                minLength={6}
              />
            </div>
          )}

          <Button type="submit" className="w-full" disabled={loading}>
            {loading ? (
              <div className="flex items-center justify-center gap-2">
                <Loader2 className="animate-spin" size={20} />
                Processing…
              </div>
            ) : isSignUp ? (
              `Sign Up ${selectedPackage === 'Pro' ? 'with Pro' : ''}`
            ) : (
              "Sign In"
            )}
          </Button>
        </form>

        {error && (
          <div className="mt-4 p-3 bg-red-50 border border-red-200 rounded-md">
            <p className="text-sm text-red-600 font-medium text-center">
              {error}
            </p>
          </div>
        )}
        {message && (
          <div className="mt-4 p-3 bg-green-50 border border-green-200 rounded-md">
            <p className="text-sm text-green-600 font-medium text-center">
              {message}
            </p>
          </div>
        )}

        <div className="mt-6 text-center">
          <button
            onClick={() => {
              setIsSignUp(!isSignUp);
              setError(null);
              setMessage(null);
              setPassword("");
              setConfirmPassword("");
              if (!isSignUp) setName("");
              setSelectedPackage('Basic');
            }}
            className="text-sm text-indigo-600 hover:underline"
          >
            {isSignUp
              ? "Already have an account? Sign In"
              : "New here? Create an account"}
          </button>
        </div>
      </div>
    </section>
  );
};

export default Auth;