import React, { useEffect, useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { supabase } from '@/components/supabaseClient';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { LogOut as Signout } from 'lucide-react';
import {
  User,
  TrendingUp,
  Lightbulb,
  Building2,
  Target,
  Heart,
  BookOpen,
  BarChart3,
  CheckCircle2,
  Calendar,
  TrendingDown,
  Goal,
  Plus,
  Crown,
} from 'lucide-react';

interface TrackerModule {
  id: string;
  title: string;
  description: string;
  icon: React.ElementType;
  gradient: string;
  stats?: {
    label: string;
    value: string;
    trend?: 'up' | 'down' | 'stable';
  };
  isActive: boolean;
  tier: 'Basic' | 'Pro' | 'Enterprise' | 'Civic';
}

interface StreakData {
  days: number;
  lastEntryDate: string;
}

interface WellnessData {
  score: number;
  mood: string;
  lastUpdated: string;
}

interface Goal {
  id: string;
  title: string;
  description: string;
  target_date: string;
  completion_percentage: number;
  created_at: string;
}

interface GoalMetrics {
  completed: number;
  pending: number;
  total: number;
  score: number;
  average: number;
}

const trackerModules: TrackerModule[] = [
  {
    id: 'pdt',
    title: 'Personal Development',
    description: 'Track mood, habits, and personal growth milestones',
    icon: User,
    gradient: 'bg-gradient-growth',
    stats: { label: 'Streak', value: '12 days', trend: 'up' },
    isActive: true,
    tier: 'Basic',
  },
  {
    id: 'cdt',
    title: 'Career Development',
    description: 'Monitor Professional growth and skill building and Innovation Management',
    icon: TrendingUp,
    gradient: 'bg-gradient-primary',
    stats: { label: 'Skills and Ideas', value: '8 active', trend: 'up' },
    isActive: false,
    tier: 'Pro',
  },
  {
    id: 'odt',
    title: 'Organizational Tools',
    description: 'Team dashboards, Remote Management, KPIs, and organizational health',
    icon: Building2,
    gradient: 'bg-gradient-hero',
    stats: { label: 'Teams', value: '5 active', trend: 'up' },
    isActive: false,
    tier: 'Enterprise',
  },
  {
    id: 'els',
    title: 'Elective Leadership Suite',
    description: 'Tools for Leadership development, Strategic Planning, and Public engagement',
    icon: Lightbulb,
    gradient: 'bg-gradient-wellness',
    stats: { label: 'Ideas', value: '23 tracked', trend: 'stable' },
    isActive: false,
    tier: 'Civic',
  },
];

const quickActions = [
  { icon: BookOpen, label: 'Journal Entry', action: 'journal' },
  { icon: Goal, label: 'Add Goal', action: 'goal' },
  { icon: CheckCircle2, label: 'Add Habits', action: 'habit' },
  { icon: TrendingDown, label: 'Budget', action: 'budget' },
];

interface DTTDashboardProps {
  onNavigate?: (view: string) => void;
}

export function IndividualDashboard({ onNavigate }: DTTDashboardProps) {
  const [userName, setUserName] = useState<string | null>(null);
  const [hasOrganization, setHasOrganization] = useState(false);
  const [packageType, setPackageType] = useState<'Basic' | 'Pro' | 'Enterprise' | 'Civic'>('Basic');
  const [isLoading, setIsLoading] = useState(true);
  const navigate = useNavigate();

  const [streakData, setStreakData] = useState<StreakData>({
    days: 0,
    lastEntryDate: ''
  });

  const [wellnessData, setWellnessData] = useState<WellnessData>({
    score: 0,
    mood: '',
    lastUpdated: ''
  });

  const [goals, setGoals] = useState<Goal[]>([]);
  const [goalMetrics, setGoalMetrics] = useState<GoalMetrics>({
    completed: 0,
    pending: 0,
    total: 0,
    score: 0,
    average: 0
  });

  // Check if user has career development access based on package_type
  const hasCareerDevelopmentAccess = (): boolean => {
    return packageType === 'Pro' || packageType === 'Enterprise' || packageType === 'Civic';
  };

  const calculateStreak = async (userId: string) => {
    try {
      const { data: entries, error } = await supabase
        .from('daily_journals')
        .select('created_at')
        .eq('user_id', userId)
        .order('created_at', { ascending: false });

      if (error) throw error;
      if (!entries || entries.length === 0) return 0;

      const dates = entries.map(e => new Date(e.created_at));
      dates.sort((a, b) => a.getTime() - b.getTime());

      let streak = 0;
      const today = new Date();
      const yesterday = new Date(today);
      yesterday.setDate(yesterday.getDate() - 1);

      const lastEntry = new Date(dates[dates.length - 1]);
      const isToday = lastEntry.toDateString() === today.toDateString();
      const isYesterday = lastEntry.toDateString() === yesterday.toDateString();

      if (!isToday && !isYesterday) return 0;

      streak = 1;
      let currentDate = isToday ? new Date(today) : new Date(yesterday);
      currentDate.setDate(currentDate.getDate() - 1);

      for (let i = dates.length - 2; i >= 0; i--) {
        const entryDate = new Date(dates[i]);
        if (entryDate.toDateString() === currentDate.toDateString()) {
          streak++;
          currentDate.setDate(currentDate.getDate() - 1);
        } else if (entryDate < currentDate) {
          break;
        }
      }
      return streak;
    } catch (error) {
      console.error('Error calculating streak:', error);
      return 0;
    }
  };

  const fetchWellnessData = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      
      if (user) {
        const { data, error } = await supabase
          .from('daily_journals')
          .select('*')
          .eq('user_id', user.id)
          .order('created_at', { ascending: false })
          .limit(1)
          .single();

        if (error) {
          console.error('Wellness data fetch error:', error);
          return;
        }  

        if (data) {
          const score = Math.min(
            10,
            Math.round(
              (data.mood === 'happy' ? 1 : 
               data.mood === 'calm' ? 0.8 : 
               data.mood === 'thoughtful' ? 0.6 : 
               data.mood === 'determined' ? 0.4 : 
               data.mood === 'low' ? 0.2 : 
               data.mood === 'tired' ? 0.1 : 0.4) * 10
            )
          );

          setWellnessData({
            score: score,
            mood: data.mood || 'neutral',
            lastUpdated: data.created_at ? new Date(data.created_at).toLocaleDateString() : ''
          });
        }
      }
    } catch (err) {
      console.error('Wellness fetch failed:', err);
    }
  };

  const handleSignOut = async () => {
    try {
      const { error } = await supabase.auth.signOut();
      if (error) throw error;
      window.location.href = "/Auth";
    } catch (err) {
      console.error("Error signing out:", err);
    }
  };

  const getWellnessRating = (score: number) => {
    if (score === 10) return 'Excellent';
    if (score >= 8.5) return 'Very Good';
    if (score >= 7) return 'Good';
    if (score >= 5) return 'Average';
    if (score >= 3) return 'Below Average';
    return 'Poor';
  };

  useEffect(() => {
    const fetchGoals = async () => {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      const { data, error } = await supabase
        .from('goals')
        .select('*')
        .eq('user_id', user.id)
        .order('created_at', { ascending: false });

      if (error) {
        console.error('Error fetching goals:', error);
        return;
      }

      if (data) {
        setGoals(data);
        const completed = data.filter(g => g.completion_percentage === 100).length;
        const pending = data.filter(g => g.completion_percentage === 0).length;
        const total = data.length;
        const score = data.reduce((sum, goal) => sum + goal.completion_percentage, 0);
        const average = total > 0 ? Math.round(score / total) : 0;

        setGoalMetrics({
          completed,
          pending,
          total,
          score,
          average
        });
      }
    };

    fetchGoals();
  }, []);

  useEffect(() => {
    const fetchUser = async () => {
      setIsLoading(true);
      try {
        await fetchWellnessData();
        const { data: { user } } = await supabase.auth.getUser();

        if (user) {
          setUserName(user.user_metadata?.name || user.user_metadata?.full_name || user.email || null);

          // Fetch user's package type from the users table
          const { data: userProfile, error: profileError } = await supabase
            .from('users')
            .select('package_type')
            .eq('id', user.id)
            .single();

          if (profileError) {
            console.error('Error fetching user profile:', profileError);
          } else if (userProfile?.package_type) {
            setPackageType(userProfile.package_type);
          } else {
            // Fallback to user metadata if users table doesn't have the data
            const metadataPackage = user.user_metadata?.package_type;
            if (metadataPackage) {
              setPackageType(metadataPackage);
            }
          }

          // Check if user has an organization
          const { data: orgData } = await supabase
            .from("organization_members")
            .select("organization_id")
            .eq("user_id", user.id)
            .maybeSingle();

          setHasOrganization(!!orgData?.organization_id);

          // Calculate streak
          const streak = await calculateStreak(user.id);
          
          // Get last journal entry date
          const { data: lastEntry } = await supabase
            .from('daily_journals')
            .select('created_at')
            .eq('user_id', user.id)
            .order('created_at', { ascending: false })
            .limit(1)
            .single();

          setStreakData({
            days: streak || 0,
            lastEntryDate: lastEntry?.created_at || ''
          });
        }
      } catch (error) {
        console.error("Error fetching user data:", error);
      } finally {
        setIsLoading(false);
      }
    };
    
    fetchUser();
  }, []);

  const handleQuickAction = (action: string) => {
    if (action === 'habit') {
      navigate('/Habits');
    } else if (action === 'journal') {
      navigate('/DailyJournal');
    } else if (action === 'goal') {
      navigate('/Goal');
    } else if (action === 'budget') {
      navigate('/BudgetTracker');
    }
  };

  const handleModuleClick = (moduleId: string) => {
    if (!moduleId) return;

    const routes: Record<string, string> = {
      'pdt': '/PersonalTracker',
      'cdt': '/CareerDevelopment',
      'odt': '/OrganizationalTools',
      'els': '/leadership-suite'
    };

    // Check access based on package type
    if (moduleId === 'cdt' && !hasCareerDevelopmentAccess()) {
      navigate('/ProUpgrade');
      return;
    }

    if ((moduleId === 'odt' || moduleId === 'els') && !hasOrganization) {
      navigate('/CreateOrganization');
      return;
    }

    if (routes[moduleId]) {
      navigate(routes[moduleId]);
    } else if (onNavigate) {
      onNavigate(moduleId);
    }
  };
  
  const userTrackerModules = trackerModules.map(module => ({
    ...module,
    isActive: (
      (module.id === 'pdt') ||
      (module.id === 'cdt' && hasCareerDevelopmentAccess()) ||
      ((module.id === 'odt' || module.id === 'els') && hasOrganization)
    )
  }));

  if (isLoading) {
    return <div className="min-h-screen flex items-center justify-center">Loading...</div>;
  }

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="border-b bg-card/50 backdrop-blur-sm sticky top-0 z-50">
        <div className="container mx-auto px-6 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-4">
              <div className="w-10 h-10 bg-gradient-hero rounded-lg flex items-center justify-center">
                <BarChart3 className="w-6 h-6 text-white" />
              </div>
              <div>
                <h1 className="text-2xl font-bold bg-gradient-hero bg-clip-text text-transparent">
                  DTT Platform
                </h1>
                <p className="text-sm text-muted-foreground">Development Tracking Tools</p>
              </div>
            </div>
            <div className="flex flex-wrap md:flex-nowrap items-center gap-3">
            <Badge
            variant={packageType === 'Pro' ? 'default' : 'secondary'}
            className={packageType === 'Pro' ? 'bg-yellow-400 text-black' : ''}
            >
            {packageType === 'Pro' && <Crown className="w-3 h-3 mr-1" />}
            {packageType} Plan
            </Badge>

            <Button
            variant="outline"
            size="sm"
            onClick={() => navigate("/CreateOrganization")}
            className="w-full sm:w-auto"
            >
            <Plus className="w-4 h-4 mr-1" />
            Create Organization
            </Button>

            {userName && hasOrganization && (
            <Button
            variant="default"
             size="sm"
            onClick={() => navigate("/OrganizationDashboard")}
            className="w-full sm:w-auto"
             >
            Go to Organization Dashboard
            </Button>
            )}

            {!hasCareerDevelopmentAccess() && (
            <Button
            variant="hero"
            size="sm"
            onClick={() => navigate("/ProUpgrade")}
            className="w-full sm:w-auto"
            >
            <Crown className="w-4 h-4 mr-1" />
            Upgrade to Pro
           </Button>
           )}

           <Button
           variant="outline"
           onClick={handleSignOut}
           className="w-full sm:w-auto"
           >
           <Signout className="mr-2 h-4 w-4" /> Sign Out
           </Button>
          </div>

          </div>
        </div>
      </div>

      {/* Main Content */}
      <div className="container mx-auto px-6 py-8">
        {/* Welcome Section */}
        <div className="mb-8 animate-fade-in">
          <h2 className="text-3xl font-semibold mb-2 text-foreground">
            Welcome back{userName ? `, ${userName}` : ''}!
          </h2>
          <p className="text-muted-foreground text-lg">
            {hasCareerDevelopmentAccess() 
              ? "Your Pro features are active! Continue your growth journey."
              : "Ready to continue your growth journey? Here's your development overview."
            }
          </p>
        </div>

        {/* Quick Actions */}
        <div className="mb-8 animate-slide-up">
          <h3 className="text-lg font-semibold mb-4 text-foreground">Quick Actions</h3>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
            {quickActions.map((action) => (
              <Card
                key={action.action}
                className="cursor-pointer hover:shadow-medium transition-all duration-300 transform hover:scale-105 border-border/50"
                onClick={() => handleQuickAction(action.action)}
              >
                <CardContent className="p-4 text-center">
                  <action.icon className="w-8 h-8 mx-auto mb-2 text-accent-bright" />
                  <p className="text-sm font-medium text-foreground">{action.label}</p>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>

        {/* Tracker Modules */}
        <div className="mb-8 animate-slide-up" style={{ animationDelay: '0.2s' }}>
          <h3 className="text-lg font-semibold mb-4 text-foreground">Development Trackers</h3>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            {userTrackerModules.map((module) => (
              <Card
                key={module.id}
                className={`cursor-pointer transition-all duration-300 hover:shadow-strong transform hover:scale-105 border-border/50 ${
                  !module.isActive ? 'opacity-60' : ''
                }`}
                onClick={() => module.isActive && handleModuleClick(module.id)}
              >
                <CardHeader className="pb-3">
                  <div className="flex items-start justify-between">
                    <div className="flex items-center space-x-3">
                      <div
                        className={`w-12 h-12 ${module.gradient} rounded-lg flex items-center justify-center shadow-medium`}
                      >
                        <module.icon className="w-6 h-6 text-white" />
                      </div>
                      <div>
                        <CardTitle className="text-lg text-foreground">{module.title}</CardTitle>
                        <Badge
                          variant={module.isActive ? 'default' : 'secondary'}
                          className="text-xs mt-1"
                        >
                          {module.tier}
                        </Badge>
                      </div>
                    </div>
                    {!module.isActive && (
                      <Button 
                        size="sm" 
                        className={
                          module.id === 'cdt' 
                            ? 'bg-yellow-400 text-black hover:bg-yellow-300' 
                            : 'bg-blue-500 text-white hover:bg-blue-600'
                        }
                        onClick={(e) => {
                          e.stopPropagation();
                          if (module.id === 'cdt') {
                            navigate('/ProUpgrade');
                          } else if (module.id === 'odt' || module.id === 'els') {
                            navigate('/CreateOrganization');
                          }
                        }}
                      >
                        {module.id === 'cdt' ? 'Upgrade to Pro' : 'Create Organization'}
                      </Button>
                    )}
                  </div>
                </CardHeader>
                <CardContent>
                  <CardDescription className="text-muted-foreground mb-3">
                    {module.description}
                  </CardDescription>
                  {module.stats && module.isActive && (
                    <div className="flex items-center justify-between pt-2 border-t border-border/50">
                      <span className="text-sm text-muted-foreground">{module.stats.label}</span>
                      <div className="flex items-center space-x-1">
                        <span className="font-medium text-foreground">{module.stats.value}</span>
                        {module.stats.trend === 'up' && (
                          <TrendingUp className="w-4 h-4 text-secondary-accent" />
                        )}
                      </div>
                    </div>
                  )}
                </CardContent>
              </Card>
            ))}
          </div>
        </div>

        {/* Today's Summary */}
        <div className="animate-slide-up" style={{ animationDelay: '0.4s' }}>
          <h3 className="text-lg font-semibold mb-4 text-foreground">Today's Summary</h3>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            <Card className="border-border/50">
              <CardHeader className="pb-3">
                <CardTitle className="text-base text-foreground flex items-center">
                  <Heart className="w-5 h-5 mr-2 text-secondary-accent" />
                  Wellness Score
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold text-foreground">{wellnessData.score}/10</div>
                <Badge variant={
                  wellnessData.score === 10 ? 'default' : 
                  wellnessData.score >= 8.5 ? 'secondary' :
                  wellnessData.score >= 7 ? 'outline' :
                  wellnessData.score >= 5 ? 'secondary' :
                  'destructive'
                } className="mt-2">
                  {getWellnessRating(wellnessData.score)}
                </Badge>
              </CardContent>
            </Card>

            <Card className="border-border/50">
              <CardHeader className="pb-3">
                <CardTitle className="text-base text-foreground flex items-center">
                  <Target className="w-5 h-5 mr-2 text-accent-bright" />
                  Goals Progress
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold text-foreground">
                  {goalMetrics.average}%
                </div>
                <div className="flex justify-between text-sm text-muted-foreground">
                  <span>Completed: {goalMetrics.completed}</span>
                  <span>Pending: {goalMetrics.pending}</span>
                  <span>Total: {goalMetrics.total}</span>
                </div>
                <div className="mt-2 w-full bg-gray-200 rounded-full h-2.5">
                  <div 
                    className="bg-green-500 h-2.5 rounded-full" 
                    style={{ width: `${goalMetrics.average}%` }}
                  ></div>
                </div>
              </CardContent>
            </Card>

            <Card className="border-border/50">
              <CardHeader className="pb-3">
                <CardTitle className="text-base text-foreground flex items-center">
                  <Calendar className="w-5 h-5 mr-2 text-primary" />
                  Active Streak
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="text-2xl font-bold text-foreground">{streakData.days} days</div>
                <p className="text-sm text-muted-foreground">
                  {streakData.days > 0 ? streakData.days > 5 ? 'Personal best!' : 'Keep going!' : 'Start your streak today!'}
                </p>
              </CardContent>
            </Card>
          </div>
        </div>

        {/* Pro Features Highlight */}
        {hasCareerDevelopmentAccess() && (
          <div className="mt-8 p-6 bg-gradient-to-r from-yellow-50 to-amber-50 border border-yellow-200 rounded-lg">
            <div className="flex items-center mb-4">
              <Crown className="w-6 h-6 text-yellow-600 mr-2" />
              <h3 className="text-lg font-semibold text-yellow-800">Pro Features Active</h3>
            </div>
            <p className="text-yellow-700 mb-3">
              Your Pro package gives you access to Career Development tools and advanced features.
            </p>
            <Button 
              variant="outline" 
              className="border-yellow-400 text-yellow-700 hover:bg-yellow-50"
              onClick={() => navigate('/CareerDevelopment')}
            >
              Explore Career Development
            </Button>
          </div>
        )}
      </div>
    </div>
  );
}