import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { supabase } from '@/components/supabaseClient';
import { Loader2, Calendar, User, Folder, AlertCircle } from 'lucide-react';

interface AddTaskFormProps {
  onSuccess: () => void;
  onCancel: () => void;
}

interface Project {
  id: string;
  name: string;
  description: string;
}

interface TeamMember {
  id: string;
  name: string;
  email: string;
  role: string;
  team_id: string;
}

interface Team {
  id: string;
  name: string;
}

export function AddTaskForm({ onSuccess, onCancel }: AddTaskFormProps) {
  const [formData, setFormData] = useState({
    name: '',
    description: '',
    due_date: '',
    assigned_to: '',
    project_id: '',
    priority: 'medium',
    status: 'pending'
  });
  const [projects, setProjects] = useState<Project[]>([]);
  const [teamMembers, setTeamMembers] = useState<TeamMember[]>([]);
  const [teams, setTeams] = useState<Team[]>([]);
  const [isLoading, setIsLoading] = useState(false);
  const [isFetching, setIsFetching] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [selectedTeam, setSelectedTeam] = useState<string>('');

  useEffect(() => {
    fetchData();
  }, []);

  useEffect(() => {
    if (selectedTeam) {
      fetchTeamMembers(selectedTeam);
    }
  }, [selectedTeam]);

  const fetchData = async () => {
    try {
      setIsFetching(true);
      setError(null);

      // Fetch all projects
      const { data: projectsData, error: projectsError } = await supabase
        .from('projects')
        .select('id, name, description')
        .order('name');

      // Fetch all teams
      const { data: teamsData, error: teamsError } = await supabase
        .from('teams')
        .select('id, name')
        .order('name');

      if (projectsError) throw projectsError;
      if (teamsError) throw teamsError;

      setProjects(projectsData || []);
      setTeams(teamsData || []);

      // Fetch team members for the first team by default
      if (teamsData && teamsData.length > 0) {
        setSelectedTeam(teamsData[0].id);
        await fetchTeamMembers(teamsData[0].id);
      }

    } catch (err) {
      console.error('Error fetching data:', err);
      setError('Failed to load data. Please try again.');
    } finally {
      setIsFetching(false);
    }
  };

  const fetchTeamMembers = async (teamId: string) => {
    try {
      const { data: membersData, error: membersError } = await supabase
        .from('team_members')
        .select('id, name, email, role, team_id')
        .eq('team_id', teamId)
        .order('name');

      if (membersError) throw membersError;
      setTeamMembers(membersData || []);
    } catch (err) {
      console.error('Error fetching team members:', err);
      setError('Failed to load team members.');
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!formData.name || !formData.due_date) {
      setError('Please fill in all required fields');
      return;
    }

    setIsLoading(true);
    setError(null);

    try {
      const taskData = {
        name: formData.name,
        description: formData.description,
        due_date: formData.due_date,
        assigned_to: formData.assigned_to || null,
        project_id: formData.project_id || null,
        priority: formData.priority,
        status: formData.status,
        created_at: new Date().toISOString()
      };

      const { error: insertError } = await supabase
        .from('tasks')
        .insert([taskData]);

      if (insertError) throw insertError;

      console.log('Task created successfully:', taskData);
      onSuccess();
      
    } catch (err) {
      console.error('Error creating task:', err);
      setError('Failed to create task. Please try again.');
    } finally {
      setIsLoading(false);
    }
  };

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
    setError(null);
  };

  const handleTeamChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    setSelectedTeam(e.target.value);
    setFormData(prev => ({ ...prev, assigned_to: '' })); // Reset assigned_to when team changes
  };

  const getMinDate = () => {
    const today = new Date();
    return today.toISOString().split('T')[0];
  };

  if (isFetching) {
    return (
      <div className="flex flex-col items-center justify-center py-8">
        <Loader2 className="w-8 h-8 animate-spin text-primary mb-4" />
        <p className="text-muted-foreground">Loading data...</p>
      </div>
    );
  }

  return (
    <form onSubmit={handleSubmit} className="space-y-6">
      <div>
        <h3 className="text-lg font-medium text-foreground mb-4">
          Create New Task
        </h3>
        
        {error && (
          <div className="bg-destructive/15 text-destructive p-3 rounded-md mb-4 flex items-center">
            <AlertCircle className="w-4 h-4 mr-2" />
            <p className="text-sm">{error}</p>
          </div>
        )}

        <div className="space-y-4">
          {/* Task Name */}
          <div>
            <label className="block text-sm font-medium text-foreground mb-2">
              Task Name *
            </label>
            <input
              type="text"
              name="name"
              required
              value={formData.name}
              onChange={handleInputChange}
              placeholder="Enter task name"
              className="w-full p-3 border border-border rounded-md bg-background text-foreground focus:ring-2 focus:ring-primary focus:border-transparent"
            />
          </div>

          {/* Description */}
          <div>
            <label className="block text-sm font-medium text-foreground mb-2">
              Description
            </label>
            <textarea
              name="description"
              value={formData.description}
              onChange={handleInputChange}
              placeholder="Describe the task..."
              rows={3}
              className="w-full p-3 border border-border rounded-md bg-background text-foreground focus:ring-2 focus:ring-primary focus:border-transparent"
            />
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            {/* Due Date */}
            <div>
              <label className="block text-sm font-medium text-foreground mb-2">
                Due Date *
              </label>
              <div className="relative">
                <Calendar className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-muted-foreground" />
                <input
                  type="date"
                  name="due_date"
                  required
                  value={formData.due_date}
                  onChange={handleInputChange}
                  min={getMinDate()}
                  className="w-full pl-10 p-3 border border-border rounded-md bg-background text-foreground focus:ring-2 focus:ring-primary focus:border-transparent"
                />
              </div>
            </div>

            {/* Priority */}
            <div>
              <label className="block text-sm font-medium text-foreground mb-2">
                Priority
              </label>
              <select
                name="priority"
                value={formData.priority}
                onChange={handleInputChange}
                className="w-full p-3 border border-border rounded-md bg-background text-foreground focus:ring-2 focus:ring-primary focus:border-transparent"
              >
                <option value="low">Low</option>
                <option value="medium">Medium</option>
                <option value="high">High</option>
                <option value="urgent">Urgent</option>
              </select>
            </div>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            {/* Team Selection */}
            <div>
              <label className="block text-sm font-medium text-foreground mb-2">
                Team
              </label>
              <div className="relative">
                <User className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-muted-foreground" />
                <select
                  value={selectedTeam}
                  onChange={handleTeamChange}
                  className="w-full pl-10 p-3 border border-border rounded-md bg-background text-foreground focus:ring-2 focus:ring-primary focus:border-transparent"
                >
                  {teams.map((team) => (
                    <option key={team.id} value={team.id}>
                      {team.name}
                    </option>
                  ))}
                </select>
              </div>
            </div>

            {/* Assign to Team Member */}
            <div>
              <label className="block text-sm font-medium text-foreground mb-2">
                Assign to
              </label>
              <div className="relative">
                <User className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-muted-foreground" />
                <select
                  name="assigned_to"
                  value={formData.assigned_to}
                  onChange={handleInputChange}
                  className="w-full pl-10 p-3 border border-border rounded-md bg-background text-foreground focus:ring-2 focus:ring-primary focus:border-transparent"
                >
                  <option value="">Unassigned</option>
                  {teamMembers.map((member) => (
                    <option key={member.id} value={member.id}>
                      {member.name} ({member.role})
                    </option>
                  ))}
                </select>
              </div>
            </div>
          </div>

          {/* Project Selection */}
          <div>
            <label className="block text-sm font-medium text-foreground mb-2">
              Related Project (Optional)
            </label>
            <div className="relative">
              <Folder className="absolute left-3 top-1/2 transform -translate-y-1/2 w-4 h-4 text-muted-foreground" />
              <select
                name="project_id"
                value={formData.project_id}
                onChange={handleInputChange}
                className="w-full pl-10 p-3 border border-border rounded-md bg-background text-foreground focus:ring-2 focus:ring-primary focus:border-transparent"
              >
                <option value="">No project</option>
                {projects.map((project) => (
                  <option key={project.id} value={project.id}>
                    {project.name}
                  </option>
                ))}
              </select>
            </div>
          </div>
        </div>
      </div>

      <div className="flex space-x-3 pt-4">
        <Button
          type="button"
          variant="outline"
          onClick={onCancel}
          disabled={isLoading}
          className="flex-1"
        >
          Cancel
        </Button>
        <Button
          type="submit"
          disabled={isLoading || !formData.name || !formData.due_date}
          className="flex-1 bg-primary hover:bg-primary/90"
        >
          {isLoading ? (
            <>
              <Loader2 className="w-4 h-4 animate-spin mr-2" />
              Creating...
            </>
          ) : (
            'Create Task'
          )}
        </Button>
      </div>
    </form>
  );
}