import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { supabase } from '@/components/supabaseClient';

interface AddProjectFormProps {
  onSuccess: () => void;
  onCancel: () => void;
}

export function AddProjectForm({ onSuccess, onCancel }: AddProjectFormProps) {
  const [formData, setFormData] = useState({
    name: '',
    description: '',
    deadline: '',
    team_id: ''
  });
  const [teams, setTeams] = useState([]);
  const [isLoading, setIsLoading] = useState(false);

  useEffect(() => {
    fetchTeams();
  }, []);

  const fetchTeams = async () => {
    const { data } = await supabase.from('teams').select('*');
    setTeams(data || []);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsLoading(true);

    try {
      const { error } = await supabase
        .from('projects')
        .insert([{ ...formData, status: 'active' }]);

      if (error) throw error;
      onSuccess();
    } catch (error) {
      console.error('Error adding project:', error);
    } finally {
      setIsLoading(false);
    }
  };

  return (
    <form onSubmit={handleSubmit} className="space-y-4">
      <div>
        <label className="block text-sm font-medium text-foreground mb-1">
          Project Name
        </label>
        <input
          type="text"
          required
          value={formData.name}
          onChange={(e) => setFormData({ ...formData, name: e.target.value })}
          className="w-full p-2 border border-border rounded-md bg-background text-foreground"
        />
      </div>
      <div>
        <label className="block text-sm font-medium text-foreground mb-1">
          Description
        </label>
        <textarea
          value={formData.description}
          onChange={(e) => setFormData({ ...formData, description: e.target.value })}
          className="w-full p-2 border border-border rounded-md bg-background text-foreground"
          rows={3}
        />
      </div>
      <div>
        <label className="block text-sm font-medium text-foreground mb-1">
          Deadline
        </label>
        <input
          type="date"
          required
          value={formData.deadline}
          onChange={(e) => setFormData({ ...formData, deadline: e.target.value })}
          className="w-full p-2 border border-border rounded-md bg-background text-foreground"
        />
      </div>
      <div>
        <label className="block text-sm font-medium text-foreground mb-1">
          Team
        </label>
        <select
          required
          value={formData.team_id}
          onChange={(e) => setFormData({ ...formData, team_id: e.target.value })}
          className="w-full p-2 border border-border rounded-md bg-background text-foreground"
        >
          <option value="">Select Team</option>
          {teams.map((team) => (
            <option key={team.id} value={team.id}>
              {team.name}
            </option>
          ))}
        </select>
      </div>
      <div className="flex space-x-3 pt-4">
        <Button type="button" variant="outline" onClick={onCancel}>
          Cancel
        </Button>
        <Button type="submit" disabled={isLoading}>
          {isLoading ? 'Creating...' : 'Create Project'}
        </Button>
      </div>
    </form>
  );
}