import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Progress } from '@/components/ui/progress';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { ArrowLeft, BookOpen, TrendingUp, Clock, Award, Star, Play, CheckCircle, Target, Users, Brain, Code, Plus, X, Edit } from 'lucide-react';
import { useNavigate } from "react-router-dom";
import { supabase } from '@/components/supabaseClient';


interface SkillDevelopmentProps {
  onBack: () => void;
}

interface Skill {
  id: string;
  name: string;
  level: number;
  category: string;
  trend: string;
  hours: number;
  description?: string;
  timeline?: string;
  modules?: string[];
  paths?: string[];
  progress?: number;
  certification?: string;
  user_id: string;
  created_at: string;
}

interface SkillFormData {
  name: string;
  level: number;
  description: string;
  timeline: string;
  modules: string;
  paths: string;
  progress: number;
  certification: string;
  category: string;
  learning_paths: string;
}

// Default skill categories structure
const defaultSkillCategories = [
  {
    id: 'technical',
    name: 'Technical Skills',
    icon: Code,
    color: 'text-blue-500',
    skills: [] as Skill[]
  },
  {
    id: 'leadership',
    name: 'Leadership',
    icon: Users,
    color: 'text-green-500',
    skills: [] as Skill[]
  },
  {
    id: 'creative',
    name: 'Creative',
    icon: Brain,
    color: 'text-purple-500',
    skills: [] as Skill[]
  }
];

export function SkillDevelopment({ onBack }: SkillDevelopmentProps, { onClose, children }: { onClose: () => void; children: React.ReactNode }) {
  const [isOpen, setIsOpen] = useState(false);
  const navigate = useNavigate();
  const [activeCategory, setActiveCategory] = useState('technical');
  const [showAddSkillForm, setShowAddSkillForm] = useState(false);
  const [editingSkill, setEditingSkill] = useState<Skill | null>(null);
  const [skillFormData, setSkillFormData] = useState<SkillFormData>({
    name: '',
    level: 0,
    description: '',
    timeline: '',
    modules: '',
    paths: '',
    progress: 0,
    certification: '',
    category: 'technical',
    learning_paths: ''
  });
  const [userSkills, setUserSkills] = useState<Skill[]>([]);
  const [loading, setLoading] = useState(true);
  const [skillCategories, setSkillCategories] = useState(defaultSkillCategories);
  const [learningPaths, setLearningPaths] = useState<any[]>([]);
  const [certifications, setCertifications] = useState<any[]>([]);
  const [recommendations, setRecommendations] = useState<any[]>([]);

  useEffect(() => {
    fetchSkills();
    fetchLearningPaths();
    fetchCertifications();
    fetchRecommendations();
  }, []);

  useEffect(() => {
    // Update skill categories with user skills
    const updatedCategories = defaultSkillCategories.map(category => ({
      ...category,
      skills: userSkills.filter(skill => skill.category === category.id)
    }));
    setSkillCategories(updatedCategories);
  }, [userSkills]);

  const fetchSkills = async () => {
    try {
      setLoading(true);
      const { data: { user } } = await supabase.auth.getUser();
      
      if (!user) {
        console.error('No user logged in');
        return;
      }

      const { data, error } = await supabase
        .from('skills')
        .select('*')
        .eq('user_id', user.id)
        .order('created_at', { ascending: false });

      if (error) {
        throw error;
      }

      setUserSkills(data || []);
    } catch (error) {
      console.error('Error fetching skills:', error);
    } finally {
      setLoading(false);
    }
  };

  const fetchLearningPaths = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      
      if (!user) return;

      const { data, error } = await supabase
        .from('skills')
        .select('*')
        .eq('user_id', user.id)
        .order('created_at', { ascending: false });

      if (error) {
        throw error;
      }

      setLearningPaths(data || []);
    } catch (error) {
      console.error('Error fetching learning paths:', error);
    }
  };

  const fetchCertifications = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      
      if (!user) return;

      const { data, error } = await supabase
        .from('skills')
        .select('*')
        .eq('user_id', user.id)
        .order('created_at', { ascending: true });

      if (error) {
        throw error;
      }

      setCertifications(data || []);
    } catch (error) {
      console.error('Error fetching certifications:', error);
    }
  };

  const fetchRecommendations = async () => {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      
      if (!user) return;

      const { data, error } = await supabase
        .from('skill_recommendations')
        .select('*')
        .eq('user_id', user.id)
        .order('priority', { ascending: false });

      if (error) {
        throw error;
      }

      setRecommendations(data || []);
    } catch (error) {
      console.error('Error fetching recommendations:', error);
    }
  };

  const getSkillColor = (level: number) => {
    if (level >= 80) return 'bg-green-500';
    if (level >= 60) return 'bg-blue-500';
    if (level >= 40) return 'bg-yellow-500';
    return 'bg-red-500';
  };

  const getDifficultyColor = (difficulty: string) => {
    switch (difficulty) {
      case 'Beginner': return 'bg-green-100 text-green-800';
      case 'Intermediate': return 'bg-yellow-100 text-yellow-800';
      case 'Advanced': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case 'High': return 'bg-red-100 text-red-800';
      case 'Medium': return 'bg-yellow-100 text-yellow-800';
      case 'Low': return 'bg-green-100 text-green-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setSkillFormData(prev => ({
      ...prev,
      [name]: name === 'level' || name === 'progress' ? Number(value) : value
    }));
  };

  const handleSelectChange = (name: string, value: string) => {
    setSkillFormData(prev => ({
      ...prev,
      [name]: name === 'level' || name === 'progress' ? Number(value) : value
    }));
  };

  const handleEditSkill = (skill: Skill) => {
    setEditingSkill(skill);
    setSkillFormData({
      name: skill.name,
      level: skill.level,
      description: skill.description || '',
      timeline: skill.timeline || '',
      modules: skill.modules?.join(', ') || '',
      paths: skill.paths?.join(', ') || '',
      progress: skill.progress || 0,
      certification: skill.certification || '',
      category: skill.category,
      learning_paths: '' // You might want to populate this if you have learning paths data
    });
    setShowAddSkillForm(true);
  };

  const handleAddSkill = async () => {
    if (!skillFormData.name) return;
    
    try {
      const { data: { user }, error: authError } = await supabase.auth.getUser();
      
      if (authError) {
        console.error('Auth error:', authError);
        throw new Error(`Authentication failed: ${authError.message}`);
      }
      
      if (!user) {
        throw new Error('No user logged in. Please sign in again.');
      }

      const modulesArray = skillFormData.modules.split(',').map(module => module.trim()).filter(module => module);
      const pathsArray = skillFormData.paths.split(',').map(path => path.trim()).filter(path => path);
      
      if (editingSkill) {
        // Update existing skill
        const { data, error } = await supabase
          .from('skills')
          .update({
            name: skillFormData.name,
            level: skillFormData.level,
            category: skillFormData.category,
            description: skillFormData.description,
            timeline: skillFormData.timeline,
            modules: modulesArray,
            paths: pathsArray,
            progress: skillFormData.progress,
            certification: skillFormData.certification,
          })
          .eq('id', editingSkill.id)
          .select();

        if (error) {
          throw error;
        }

        if (data && data.length > 0) {
          setUserSkills(prev => prev.map(skill => 
            skill.id === editingSkill.id ? data[0] : skill
          ));
        }
      } else {
        // Add new skill
        const { data, error } = await supabase
          .from('skills')
          .insert([
            {
              name: skillFormData.name,
              level: skillFormData.level,
              category: skillFormData.category,
              trend: 'up',
              hours: 0,
              description: skillFormData.description,
              timeline: skillFormData.timeline,
              modules: modulesArray,
              paths: pathsArray,
              progress: skillFormData.progress,
              certification: skillFormData.certification,
              user_id: user.id
            }
          ])
          .select();

        if (error) {
          throw error;
        }

        if (data && data.length > 0) {
          setUserSkills(prev => [data[0], ...prev]);
        }
      }
      
      setSkillFormData({
        name: '',
        level: 0,
        description: '',
        timeline: '',
        modules: '',
        paths: '',
        progress: 0,
        certification: '',
        category: 'technical',
        learning_paths: ''
      });
      setEditingSkill(null);
      setShowAddSkillForm(false);
      
    } catch (error) {
      console.error('Error saving skill:', error);
    }
  };

  const handleDeleteSkill = async (skillId: string) => {
    try {
      const { error } = await supabase
        .from('skills')
        .delete()
        .eq('id', skillId);

      if (error) {
        throw error;
      }

      setUserSkills(prev => prev.filter(skill => skill.id !== skillId));
    } catch (error) {
      console.error('Error deleting skill:', error);
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-background flex items-center justify-center">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-primary mx-auto"></div>
          <p className="mt-4 text-muted-foreground">Loading skills...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="border-b bg-card/50 backdrop-blur-sm sticky top-0 z-50">
        <div className="container mx-auto px-6 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-4">
              <Button variant="ghost" size="sm"  onClick={() => navigate("/CareerDevelopment")}
                className="w-4 h-4 mr-2">
                <ArrowLeft className="w-4 h-4" />
              </Button>
              <div className="w-10 h-10 rounded-lg bg-gradient-primary flex items-center justify-center">
                <BookOpen className="w-6 h-6 text-white" />
              </div>
              <div>
                <h1 className="text-2xl font-bold bg-gradient-hero bg-clip-text text-transparent">
                  Skill Development
                </h1>
                <p className="text-sm text-muted-foreground">Track and develop your professional skills</p>
              </div>
            </div>
            <div className="flex items-center space-x-2">
              <Badge variant="secondary" className="bg-violet-800 text-white">Active Learning</Badge>
              <Button onClick={() => {
                setEditingSkill(null);
                setShowAddSkillForm(true);
              }}>
                <Plus className="w-4 h-4 mr-2" />
                Add Skill
              </Button>
            </div>
          </div>
        </div>
      </div>

      {/* Add/Edit Skill Modal */}
      {showAddSkillForm && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4 overflow-auto"
        onClick={onClose} // closes modal when background is clicked
        >

          <Card className="w-full max-w-md"
          onClick={(e) => e.stopPropagation()} // prevent closing when clicking inside modal content
           > 
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle>{editingSkill ? 'Edit Skill' : 'Add New Skill'}</CardTitle>
                <Button variant="ghost" size="sm" onClick={() => {
                  setShowAddSkillForm(false);
                  setEditingSkill(null);
                }}>
                  <div
                  className="bg-white p-6 rounded-2xl shadow-lg"
                  
                  >
                 Close
                 </div>
                </Button>
              </div>
              <CardDescription>Track your skill development progress</CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <label className="text-sm font-medium">Skill Name</label>
                <Input
                  name="name"
                  value={skillFormData.name}
                  onChange={handleInputChange}
                  placeholder="e.g., TypeScript, Project Management"
                />
              </div>
              
              <div className="space-y-2">
                <label className="text-sm font-medium">Skill Level (0-100)</label>
                <Input
                  name="level"
                  type="number"
                  min="0"
                  max="100"
                  value={skillFormData.level}
                  onChange={handleInputChange}
                />
              </div>
              
              <div className="space-y-2">
                <label className="text-sm font-medium">Category</label>
                <Select
                  value={skillFormData.category}
                  onValueChange={(value) => handleSelectChange('category', value)}
                >
                  <SelectTrigger>
                    <SelectValue placeholder="Select category" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="technical">Technical Skills</SelectItem>
                    <SelectItem value="leadership">Leadership</SelectItem>
                    <SelectItem value="creative">Creative</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              
              <div className="space-y-2">
                <label className="text-sm font-medium">Description</label>
                <Textarea
                  name="description"
                  value={skillFormData.description}
                  onChange={handleInputChange}
                  placeholder="Describe what this skill entails..."
                />
              </div>
              
              <div className="space-y-2">
                <label className="text-sm font-medium">Timeline</label>
                <Input
                  name="timeline"
                  value={skillFormData.timeline}
                  onChange={handleInputChange}
                  placeholder="e.g., 3 months, Q2 2024"
                />
              </div>
              
              <div className="space-y-2">
                <label className="text-sm font-medium">Modules (comma separated)</label>
                <Input
                  name="modules"
                  value={skillFormData.modules}
                  onChange={handleInputChange}
                  placeholder="e.g., Basics, Advanced Concepts, Practical Applications"
                />
              </div>
              
              <div className="space-y-2">
                <label className="text-sm font-medium">Progress (0-100)</label>
                <Input
                  name="progress"
                  type="number"
                  min="0"
                  max="100"
                  value={skillFormData.progress}
                  onChange={handleInputChange}
                />
              </div>
              
              <div className="space-y-2">
                <label className="text-sm font-medium">Certification to be Obtained</label>
                <Input
                  name="certification"
                  value={skillFormData.certification}
                  onChange={handleInputChange}
                  placeholder="e.g., AWS Certified Developer, PMP"
                />
              </div>
              
              <div className="space-y-2">
                <label className="text-sm font-medium">Learning Paths (comma separated)</label>
                <Input
                  name="paths"
                  value={skillFormData.paths}
                  onChange={handleInputChange}
                  placeholder="e.g., Frontend Development, Cloud Certification"
                />
              </div>
              
              <Button onClick={handleAddSkill} className="w-full">
                {editingSkill ? 'Update Skill' : 'Add Skill'}
              </Button>
            
              <Button onClick={() => {
              setShowAddSkillForm(false);
              setEditingSkill(null);
              }}
              variant="outline"
              className="w-full"
               >
              Cancel
              </Button>
            </CardContent>
          </Card>
        </div>
      )}

      {/* Main Content */}
      <div className="container mx-auto px-6 py-8">
        <Tabs defaultValue="skills" className="w-full">
          <TabsList className="grid w-full grid-cols-4">
            <TabsTrigger value="skills">Skills Overview</TabsTrigger>
            <TabsTrigger value="paths">Learning Paths</TabsTrigger>
            <TabsTrigger value="certifications">Certifications</TabsTrigger>
            <TabsTrigger value="recommendations">Recommendations</TabsTrigger>
          </TabsList>

          {/* Skills Overview Tab */}
          <TabsContent value="skills" className="space-y-6">
            {/* Skill Categories */}
            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
              {skillCategories.map((category) => (
                <Card 
                  key={category.id}
                  className={`cursor-pointer transition-all duration-300 hover:shadow-strong ${
                    activeCategory === category.id ? 'ring-2 ring-primary' : ''
                  }`}
                  onClick={() => setActiveCategory(category.id)}
                >
                  <CardHeader className="pb-3">
                    <CardTitle className="flex items-center text-lg">
                      <category.icon className={`w-5 h-5 mr-2 ${category.color}`} />
                      {category.name}
                    </CardTitle>
                    <CardDescription>
                      {category.skills.length} skills tracked
                    </CardDescription>
                  </CardHeader>
                  <CardContent>
                    <div className="space-y-2">
                      {category.skills.slice(0, 2).map((skill) => (
                        <div key={skill.id} className="flex items-center justify-between">
                          <span className="text-sm">{skill.name}</span>
                          <span className="text-sm font-medium">{skill.level}%</span>
                        </div>
                      ))}
                      {category.skills.length === 0 && (
                        <p className="text-sm text-muted-foreground">No skills added yet</p>
                      )}
                    </div>
                  </CardContent>
                </Card>
              ))}
            </div>

            {/* Detailed Skills View */}
            {activeCategory && (
              <Card>
                <CardHeader>
                  <CardTitle>
                    {skillCategories.find(cat => cat.id === activeCategory)?.name} Skills
                  </CardTitle>
                  <CardDescription>
                    Detailed breakdown of your skill levels and progress
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-6">
                    {skillCategories
                      .find(cat => cat.id === activeCategory)
                      ?.skills.map((skill) => (
                      <div key={skill.id} className="space-y-2">
                        <div className="flex items-center justify-between">
                          <div className="flex items-center space-x-2">
                            <h4 className="font-medium">{skill.name}</h4>
                            <Badge variant="outline" className="text-xs">
                              {skill.category}
                            </Badge>
                            {skill.trend === 'up' && (
                              <TrendingUp className="w-4 h-4 text-green-500" />
                            )}
                          </div>
                          <div className="flex items-center space-x-2">
                            <span className="text-sm text-muted-foreground">
                              {skill.hours}h practiced
                            </span>
                            <span className="font-bold">{skill.level}%</span>
                          </div>
                        </div>
                        <Progress value={skill.level} className="h-2" />
                        
                        <div className="flex items-center space-x-2 mt-2">
                          <Button 
                            size="sm" 
                            variant="outline"
                            onClick={() => handleEditSkill(skill)}
                          >
                            <Edit className="w-3 h-3 mr-1" />
                            Edit
                          </Button>
                          <Button 
                            variant="ghost" 
                            size="sm" 
                            className="h-8 w-8 p-0 text-destructive"
                            onClick={() => handleDeleteSkill(skill.id)}
                          >
                            <X className="w-4 h-4" />
                          </Button>
                        </div>
                        
                        {/* Additional skill details for user-added skills */}
                        {(skill.description || skill.timeline || skill.modules || skill.certification) && (
                          <div className="mt-2 p-3 bg-muted rounded-md text-sm">
                            {skill.description && (
                              <p className="mb-2">{skill.description}</p>
                            )}
                            <div className="grid grid-cols-2 gap-2">
                              {skill.timeline && (
                                <div>
                                  <span className="font-medium">Timeline:</span> {skill.timeline}
                                </div>
                              )}
                              {skill.progress !== undefined && (
                                <div>
                                  <span className="font-medium">Progress:</span> {skill.progress}%
                                </div>
                              )}
                              {skill.certification && (
                                <div>
                                  <span className="font-medium">Certification:</span> {skill.certification}
                                </div>
                              )}
                              {skill.modules && skill.modules.length > 0 && (
                                <div className="col-span-2">
                                  <span className="font-medium">Modules:</span> {skill.modules.join(', ')}
                                </div>
                              )}
                            </div>
                          </div>
                        )}
                      </div>
                    ))}
                    {skillCategories.find(cat => cat.id === activeCategory)?.skills.length === 0 && (
                      <div className="text-center py-8">
                        <BookOpen className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
                        <h3 className="text-lg font-medium mb-2">No skills yet</h3>
                        <p className="text-muted-foreground mb-4">
                          Add your first skill to start tracking your development progress
                        </p>
                        <Button onClick={() => {
                          setEditingSkill(null);
                          setShowAddSkillForm(true);
                        }}>
                          <Plus className="w-4 h-4 mr-2" />
                          Add Skill
                        </Button>
                      </div>
                    )}
                  </div>
                </CardContent>
              </Card>
            )}
          </TabsContent>

          {/* Learning Paths Tab */}
          <TabsContent value="paths" className="space-y-6">
            {learningPaths.length > 0 ? (
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {learningPaths.map((path) => (
                  <Card key={path.id} className="hover:shadow-strong transition-all duration-300">
                    <CardHeader>
                      <div className="flex items-center justify-between">
                        <CardTitle className="text-lg">{path.title}</CardTitle>
                        <Badge 
                          variant="outline" 
                          className={getDifficultyColor(path.difficulty)}
                        >
                          {path.difficulty}
                        </Badge>
                      </div>
                      <CardDescription>{path.description}</CardDescription>
                    </CardHeader>
                    <CardContent className="space-y-4">
                      <div className="flex items-center justify-between text-sm">
                        <span className="flex items-center">
                          <Clock className="w-4 h-4 mr-1" />
                          {path.duration}
                        </span>
                        <span>{path.timeline}</span>
                      </div>
                      
                      <div>
                        <div className="flex items-center justify-between mb-1">
                          <span className="text-sm font-medium">Progress</span>
                          <span className="text-sm">{path.progress}%</span>
                        </div>
                        <Progress value={path.progress} className="h-2" />
                      </div>

                      <div className="flex flex-wrap gap-1">
                        {path.paths?.slice(0, 4).map((skill: string) => (
                          <Badge key={skill} variant="secondary" className="text-xs">
                            {skill}
                          </Badge>
                        ))}
                        {path.skills?.length > 3 && (
                          <Badge variant="secondary" className="text-xs">
                            +{path.skills.length - 3} more
                          </Badge>
                        )}
                      </div>

                      <Button onClick={() => {  
                          setEditingSkill(!null);
                          setShowAddSkillForm(true);
                          handleEditSkill(skill);
                        }}
                      className="w-full" size="sm">
                        <Play className="w-4 h-4 mr-2" />
                        Continue Learning
                      </Button>

                    </CardContent>
                  </Card>
                ))}
              </div>
            ) : (
              <Card>
                <CardHeader>
                  <CardTitle>Learning Paths</CardTitle>
                  <CardDescription>
                    Your personalized learning journeys will appear here
                  </CardDescription>
                </CardHeader>
                <CardContent className="text-center py-12">
                  <Target className="w-16 h-16 text-muted-foreground mx-auto mb-4" />
                  <h3 className="text-lg font-medium mb-2">No learning paths yet</h3>
                  <p className="text-muted-foreground">
                    Learning paths will be generated based on your skills and goals
                  </p>
                </CardContent>
              </Card>
            )}
          </TabsContent>

          {/* Certifications Tab */}
          <TabsContent value="certifications" className="space-y-6">
            {certifications.length > 0 ? (
              <Card>
                <CardHeader>
                  <CardTitle>Upcoming Certifications</CardTitle>
                  <CardDescription>
                    Track your certification preparation and exam schedules
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-6">
                    {certifications.map((skills) => (
                      <div key={skills.id} className="p-4 border rounded-lg space-y-3">
                        <div className="flex items-center justify-between">
                          <div>
                            
                          <h4 className="font-medium">Certification: {skills.certification}</h4>
                  
                            <h4 className="font-semibold">Skill: {skills.name}</h4>
                            <p className="text-sm text-muted-foreground">{skills.provider}</p>
                          </div>
                          <div className="text-right">
                            <p className="text-sm font-medium">{skills.exam_date}</p>
                            <p className="text-xs text-muted-foreground">Exam Date</p>
                          </div>
                        </div>
                        
                        <div>
                          <div className="flex items-center justify-between mb-1">
                            <span className="text-sm">Preparation Progress</span>
                            <span className="text-sm">{skills.progress}%</span>
                          </div>
                          <Progress value={skills.preparation_progress} className="h-2" />
                        </div>
                        
                        <div className="flex items-center justify-between text-sm text-muted-foreground">
                          <span>Estimated study time: {skills.estimated_hours}h</span>
                          <Button variant="outline" size="sm">
                            Study Plan
                          </Button>
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            ) : (
              <Card>
                <CardHeader>
                  <CardTitle>Certifications</CardTitle>
                  <CardDescription>
                    Track your professional certifications and exam schedules
                  </CardDescription>
                </CardHeader>
                <CardContent className="text-center py-12">
                  
                  <Award className="w-16 h-16 text-muted-foreground mx-auto mb-4" />
                  <h3 className="text-lg font-medium mb-2">No certifications yet</h3>
                  <p className="text-muted-foreground">
                    Add certifications you're working towards to track your progress
                  </p>
                  
                </CardContent>
              </Card>
            )}
          </TabsContent>

          {/* Recommendations Tab */}
          <TabsContent value="recommendations" className="space-y-6">
            {recommendations.length > 0 ? (
              <Card>
                <CardHeader>
                  <CardTitle>Skill Recommendations</CardTitle>
                  <CardDescription>
                    AI-powered suggestions based on your current skills and career goals
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-4">
                    {recommendations.map((rec) => (
                      <div key={rec.id} className="p-4 border rounded-lg">
                        <div className="flex items-center justify-between mb-2">
                          <h4 className="font-semibold">{rec.skill}</h4>
                          <div className="flex space-x-2">
                            <Badge 
                              variant="outline" 
                              className={getPriorityColor(rec.priority)}
                            >
                              {rec.priority} Priority
                            </Badge>
                            <Badge 
                              variant="outline"
                              className={getDifficultyColor(rec.difficulty)}
                            >
                              {rec.difficulty}
                            </Badge>
                          </div>
                        </div>
                        <p className="text-sm text-muted-foreground mb-3">{rec.reason}</p>
                        <div className="flex items-center justify-between">
                          <span className="text-sm">
                            <Clock className="w-4 h-4 inline mr-1" />
                            {rec.estimated_time}
                          </span>
                          <Button variant="outline" size="sm">
                            <Target className="w-4 h-4 mr-2" />
                            Add to Goals
                          </Button>
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            ) : (
              <Card>
                <CardHeader>
                  <CardTitle>Recommendations</CardTitle>
                  <CardDescription>
                    Personalized skill recommendations will appear here
                  </CardDescription>
                </CardHeader>
                <CardContent className="text-center py-12">
                  <Star className="w-16 h-16 text-muted-foreground mx-auto mb-4" />
                  <h3 className="text-lg font-medium mb-2">No recommendations yet</h3>
                  <p className="text-muted-foreground">
                    Recommendations will be generated as you add more skills to your profile
                  </p>
                </CardContent>
              </Card>
            )}
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
}