import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Progress } from '@/components/ui/progress';
import { useNavigate } from 'react-router-dom';
import { Badge } from '@/components/ui/badge';
import { 
  Smile, 
  Meh, 
  Frown, 
  Target, 
  CheckCircle2, 
  Circle,
  TrendingUp,
  BookOpen,
  Calendar,
  Plus,
  ArrowLeft,
  Heart,
  Star
} from 'lucide-react';
import { supabase } from '@/components/supabaseClient';

interface MoodEntry {
  emoji: string;
  label: string;
  value: number;
}

interface Goal {
  id: string;
  title: string;
  progress: number;
  target: number;
  category: string;
  targetDate: string;
  user_id: string;
  created_at: string;
  completion_percentage: number;
  description?: string;
  status: string;
  target_date: string;  
}

interface Habit {
  id: string;
  name: string;
  description?: string;
  user_id: string;
  created_at: string;
  completed: boolean;
  streak: number;
}

interface MoodData {
  mood: string;
  created_at: string;
}

const moodOptions: MoodEntry[] = [
  { emoji: '😊', label: 'Great', value: 5 },
  { emoji: '🙂', label: 'Good', value: 4 },
  { emoji: '😐', label: 'Okay', value: 3 },
  { emoji: '😔', label: 'Low', value: 2 },
  { emoji: '😞', label: 'Difficult', value: 1 }
];

interface PersonalTrackerProps {
  onBack: () => void;
}

export function PersonalTracker({ onBack }: PersonalTrackerProps) {
  const [selectedMood, setSelectedMood] = useState<number | null>(null);
  const [habits, setHabits] = useState<Habit[]>([]);
  const [goals, setGoals] = useState<Goal[]>([]);
  const [todaysMood, setTodaysMood] = useState<string | null>(null);
  const [isLoading, setIsLoading] = useState(true);
  const [userId, setUserId] = useState<string | null>(null);
  const [editId, setEditId] = useState<string | null>(null);
  const navigate = useNavigate();

  const fetchGoals = async () => {
    if (!userId) return;
    
    try {
      const { data: goalsData, error: goalsError } = await supabase
        .from("goals")
        .select("*")
        .eq("user_id", userId)
        .order("created_at", { ascending: true });

      if (goalsError) throw goalsError;
      setGoals(goalsData || []);
    } catch (error) {
      console.error("Error fetching goals:", error);
    }
  };

  const handleEdit = (goal: Goal) => {
    // Navigate to edit page or open modal with goal data
    navigate('/Goal', { state: { editGoal: goal } });
  };

  const handleDelete = async (id: string) => {
    try {
      const { error } = await supabase
        .from("goals")
        .delete()
        .eq("id", id);

      if (error) throw error;
      
      // Refresh goals list
      fetchGoals();
    } catch (error) {
      console.error("Error deleting goal:", error);
      alert("Failed to delete goal");
    }
  };

  const markAsDone = async (id: string) => {
    try {
      const { error } = await supabase
        .from("goals")
        .update({ 
          status: 'Completed', 
          completion_percentage: 100 
        })
        .eq("id", id);

      if (error) throw error;
      
      // Refresh goals list
      fetchGoals();
    } catch (error) {
      console.error("Error marking goal as done:", error);
      alert("Failed to update goal");
    }
  };

  useEffect(() => {
    const fetchData = async () => {
      setIsLoading(true);
      try {
        // Get current user
        const { data: { user }, error: userError } = await supabase.auth.getUser();
        if (userError) throw userError;
        setUserId(user?.id || null);

        if (user?.id) {
          // Fetch today's mood from daily_journals
          const today = new Date();
          const todayUTC = new Date(Date.UTC(today.getFullYear(), today.getMonth(), today.getDate()));
          const tomorrowUTC = new Date(todayUTC);
          tomorrowUTC.setDate(tomorrowUTC.getDate() + 1);

          const { data: moodData, error: moodError } = await supabase
            .from("daily_journals")
            .select("mood, created_at")
            .eq("user_id", user.id)
            .gte("created_at", todayUTC.toISOString())
            .lt("created_at", tomorrowUTC.toISOString())
            .single();

          if (!moodError && moodData) {
            setTodaysMood(moodData.mood);
            
            // Set the selected mood based on the stored value
            const moodValue = moodOptions.find(m => m.label.toLowerCase() === moodData.mood)?.value || null;
            setSelectedMood(moodValue);
          }

          // Fetch habits
          const { data: habitsData, error: habitsError } = await supabase
            .from("habits")
            .select("*")
            .eq("user_id", user.id)
            .order("created_at", { ascending: true });

          if (habitsError) throw habitsError;

          // Fetch habit completions for today
          const todayDateString = today.toISOString().split('T')[0];
          const { data: completionsData, error: completionsError } = await supabase
            .from("habit_completions")
            .select("habit_id")
            .eq("user_id", user.id)
            .eq("completed_date", todayDateString);

          if (!completionsError) {
            const completedHabitIds = completionsData?.map(c => c.habit_id) || [];
            
            // Add completion status and mock streak data to habits
            const habitsWithStatus = (habitsData || []).map(habit => ({
              ...habit,
              completed: completedHabitIds.includes(habit.id),
              streak: Math.floor(Math.random() * 20) + 1 // Mock streak data
            }));
            
            setHabits(habitsWithStatus);
          } else {
            setHabits((habitsData || []).map(habit => ({
              ...habit,
              completed: false,
              streak: Math.floor(Math.random() * 20) + 1
            })));
          }

          // Fetch goals
          await fetchGoals();
        }
      } catch (error) {
        console.error("Error fetching data:", error);
      } finally {
        setIsLoading(false);
      }
    };

    fetchData();
  }, [userId]); // Added userId dependency

  const toggleHabit = async (habitId: string) => {
    if (!userId) return;

    const habit = habits.find(h => h.id === habitId);
    if (!habit) return;

    const today = new Date();
    const todayDateString = today.toISOString().split('T')[0];
    const isCurrentlyCompleted = habit.completed;

    try {
      if (isCurrentlyCompleted) {
        // Remove completion
        const { error } = await supabase
          .from("habit_completions")
          .delete()
          .eq("habit_id", habitId)
          .eq("user_id", userId)
          .eq("completed_date", todayDateString);

        if (error) throw error;
      } else {
        // Add completion
        const { error } = await supabase
          .from("habit_completions")
          .insert([
            {
              user_id: userId,
              habit_id: habitId,
              completed_date: todayDateString,
            },
          ]);

        if (error) throw error;
      }

      // Update local state
      setHabits(habits.map(h => 
        h.id === habitId 
          ? { ...h, completed: !h.completed }
          : h
      ));
    } catch (error) {
      console.error("Error toggling habit completion:", error);
    }
  };

  const getProgressColor = (progress: number, target: number) => {
    const percentage = (progress / target) * 100;
    if (percentage >= 90) return 'bg-secondary-accent';
    if (percentage >= 70) return 'bg-accent-bright';
    return 'bg-primary';
  };

  const completedHabits = habits.filter(h => h.completed).length;
  const totalHabits = habits.length;
  const habitProgress = totalHabits > 0 ? (completedHabits / totalHabits) * 100 : 0;

  if (isLoading) {
    return (
      <div className="min-h-screen bg-background flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-indigo-500"></div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="border-b bg-card/50 backdrop-blur-sm sticky top-0 z-50">
        <div className="container mx-auto px-6 py-4">
          <div className="flex items-center space-x-4">
            <Button variant="ghost" size="icon" onClick={()=>navigate(-1)}
              className="w-5 h-5">
              <ArrowLeft></ArrowLeft>
            </Button>
            <div className="w-10 h-10 bg-gradient-growth rounded-lg flex items-center justify-center">
              <Target className="w-6 h-6 text-white" />
            </div>
            <div>
              <h1 className="text-2xl font-bold text-foreground">Personal Development Tracker</h1>
              <p className="text-sm text-muted-foreground">Track your daily progress and growth</p>
            </div>
          </div>
        </div>
      </div>

      <div className="container mx-auto px-6 py-8">
        {/* Daily Check-in */}
        <div className="mb-8 animate-fade-in">
          <h2 className="text-xl font-semibold mb-4 text-foreground">Daily Check-in</h2>
          <Card className="border-border/50">
            <CardHeader>
              <CardTitle className="text-lg text-foreground">How are you feeling today?</CardTitle>
              <CardDescription>
                {todaysMood ? (
                  <span className="text-green-600">You've already logged your mood today: {todaysMood}</span>
                ) : (
                  "Track your daily mood to identify patterns"
                )}
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="flex space-x-4 mb-4">
                {moodOptions.map((mood) => (
                  <button
                    key={mood.value}
                    onClick={() => setSelectedMood(mood.value)}
                    className={`p-4 rounded-lg border-2 transition-all duration-300 hover:scale-110 ${
                      selectedMood === mood.value
                        ? 'border-primary shadow-medium bg-accent'
                        : 'border-border hover:border-accent-bright'
                    }`}
                  >
                    <div className="text-2xl mb-1">{mood.emoji}</div>
                    <div className="text-xs text-muted-foreground">{mood.label}</div>
                  </button>
                ))}
              </div>
              {selectedMood && (
                <div className="p-3 bg-accent rounded-lg">
                  <p className="text-sm text-accent-foreground">
                    {todaysMood ? "Mood updated!" : "Mood logged!"} Your 7-day average is trending positive 📈
                  </p>
                </div>
              )}
            </CardContent>
          </Card>
        </div>

        {/* Today's Habits */}
        <div className="mb-8 animate-slide-up">
          <div className="flex items-center justify-between mb-4">
            <h2 className="text-xl font-semibold text-foreground">Today's Habits</h2>
            <Badge variant="secondary">
              {completedHabits}/{totalHabits} completed
            </Badge>
          </div>
          
          {habits.length === 0 ? (
            <Card className="border-border/50 p-6 text-center">
              <CardContent>
                <Target className="h-12 w-12 mx-auto text-gray-400 mb-4" />
                <h3 className="text-lg font-semibold text-gray-600 mb-2">No habits yet</h3>
                <p className="text-gray-500 mb-4">Add your first habit to start tracking!</p>
                <Button onClick={() => navigate('/habits')}>
                  <Plus className="w-4 h-4 mr-2" />
                  Create Habits
                </Button>
              </CardContent>
            </Card>
          ) : (
            <>
              <Card className="border-border/50 mb-4">
                <CardContent className="p-4">
                  <div className="flex items-center justify-between mb-2">
                    <span className="text-sm font-medium text-foreground">Daily Progress</span>
                    <span className="text-sm text-muted-foreground">{Math.round(habitProgress)}%</span>
                  </div>
                  <Progress value={habitProgress} className="h-2" />
                </CardContent>
              </Card>

              <div className="grid gap-3">
                {habits.map((habit) => (
                  <Card 
                    key={habit.id}
                    className={`cursor-pointer transition-all duration-300 border-border/50 ${
                      habit.completed 
                        ? 'bg-accent/20 border-secondary-accent/30' 
                        : 'hover:shadow-soft'
                    }`}
                    onClick={() => toggleHabit(habit.id)}
                  >
                    <CardContent className="p-4">
                      <div className="flex items-center justify-between">
                        <div className="flex items-center space-x-3">
                          {habit.completed ? (
                            <CheckCircle2 className="w-6 h-6 text-secondary-accent" />
                          ) : (
                            <Circle className="w-6 h-6 text-muted-foreground" />
                          )}
                          <div>
                            <p className={`font-medium ${
                              habit.completed 
                                ? 'text-secondary-accent' 
                                : 'text-foreground'
                            }`}>
                              {habit.name}
                            </p>
                            {habit.description && (
                              <p className="text-sm text-muted-foreground">
                                {habit.description}
                              </p>
                            )}
                            <p className="text-sm text-muted-foreground">
                              {habit.streak} day streak
                            </p>
                          </div>
                        </div>
                        <div className="text-right">
                          <div className="text-sm text-muted-foreground">Streak</div>
                          <div className="font-bold text-foreground">{habit.streak}</div>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </>
          )}
        </div>

        {/* Goals Progress */}
        <div className="mb-8 animate-slide-up" style={{ animationDelay: '0.2s' }}>
          <div className="flex items-center justify-between mb-4">
            <h2 className="text-xl font-semibold text-foreground">Goals Progress</h2>
            <Button variant="outline" size="sm" onClick={() => navigate('/Goal')}>
              <Plus className="w-4 h-4 mr-1" />
              Add Goal
            </Button>
          </div>
          
          {goals.length === 0 ? (
            <Card className="border-border/50 p-6 text-center">
              <CardContent>
                <Star className="h-12 w-12 mx-auto text-gray-400 mb-4" />
                <h3 className="text-lg font-semibold text-gray-600 mb-2">No goals yet</h3>
                <p className="text-gray-500 mb-4">Set your first goal to start achieving!</p>
                <Button onClick={() => navigate('/Goal')}>
                  <Plus className="w-4 h-4 mr-2" />
                  Create Goals
                </Button>
              </CardContent>
            </Card>
          ) : (
            <div className="grid gap-4">
              {goals.map((goal) => {
                const percentage = goal.completion_percentage;
                return (
                  <Card key={goal.id} className="border-border/50">
                    <CardContent className="p-6">
                      <div className="flex items-start justify-between mb-3">
                        <div>
                          <h3 className="font-medium text-foreground mb-1">{goal.title}</h3>
                          <Badge variant="secondary" className="text-xs">
                            {goal.category || 'General'}
                          </Badge>
                        </div>
                        <div className="text-right">
                          <div className="text-sm text-muted-foreground">Status</div>
                          <div className="font-bold text-foreground">
                            {goal.status} | {goal.completion_percentage}%
                          </div>
                        </div>
                      </div>
                      
                      <div className="mb-3">
                        <div className="flex items-center justify-between mb-1">
                          <span className="text-sm text-muted-foreground">
                            {Math.round(percentage)}% complete
                          </span>
                          <span className="text-sm text-muted-foreground">
                            Due: {goal.target_date || 'No deadline'}
                          </span>
                        </div>
                        <Progress value={percentage} className="h-2" />
                      </div>
                      
                      {percentage >= 90 && percentage < 100 && (
                        <div className="flex items-center text-secondary-accent text-sm mb-3">
                          <TrendingUp className="w-4 h-4 mr-1" />
                          Almost there! Keep it up!
                        </div>
                      )}
                      
                      {goal.description && (
                        <p className="text-muted-foreground mb-2 text-sm">{goal.description}</p>
                      )}
                      
                      <div className="flex gap-2 flex-wrap">
                        <Button size="sm" onClick={() => handleEdit(goal)}>Edit</Button>
                        <Button size="sm" variant="destructive" onClick={() => handleDelete(goal.id)}>
                          Delete
                        </Button>
                        {goal.status !== 'Completed' && (
                          <Button size="sm" onClick={() => markAsDone(goal.id)}>
                            Mark as Done
                          </Button>
                        )}
                      </div>
                    </CardContent>
                  </Card>
                );
              })}
            </div>
          )}
        </div>

        {/* Quick Actions */}
        <div className="animate-slide-up" style={{ animationDelay: '0.4s' }}>
          <h2 className="text-xl font-semibold mb-4 text-foreground">Quick Actions</h2>
          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-6 gap-4">
            <Card 
              className="cursor-pointer hover:shadow-medium transition-all duration-300 transform hover:scale-105 border-border/50"
              onClick={() => navigate('/DailyJournal')}
            >
              <CardContent className="p-4 text-center">
                
                <div className="w-12 h-12 mx-auto mb-2 bg-gradient-primary rounded-lg flex items-center justify-center">
                  <BookOpen className="w-6 h-6 text-white" />
                </div>
                <p className="text-sm font-medium text-foreground mb-1">Journal Entry</p>
                <p className="text-xs text-muted-foreground">Write about your day</p>
                
              </CardContent>
            </Card>
            
            <Card 
              className="cursor-pointer hover:shadow-medium transition-all duration-300 transform hover:scale-105 border-border/50"
              onClick={() => navigate('/Goal')}
            >
              <CardContent className="p-4 text-center">
                <div className="w-12 h-12 mx-auto mb-2 bg-gradient-growth rounded-lg flex items-center justify-center">
                  <Target className="w-6 h-6 text-white" />
                </div>
                <p className="text-sm font-medium text-foreground mb-1">Add Goal</p>
                <p className="text-xs text-muted-foreground">Add a New Goal</p>
              </CardContent>
            </Card>
            
            <Card 
              className="cursor-pointer hover:shadow-medium transition-all duration-300 transform hover:scale-105 border-border/50"
              onClick={() => navigate('/habits')}
            >
              <CardContent className="p-4 text-center">
                <div className="w-12 h-12 mx-auto mb-2 bg-gradient-hero rounded-lg flex items-center justify-center">
                  <Calendar className="w-6 h-6 text-white" />
                </div>
                <p className="text-sm font-medium text-foreground mb-1">Habits</p>
                <p className="text-xs text-muted-foreground">Set your Daily Habits</p>
              </CardContent>
            </Card>
            
            <Card className="cursor-pointer hover:shadow-medium transition-all duration-300 transform hover:scale-105 border-border/50">
              <CardContent className="p-4 text-center">
                <div className="w-12 h-12 mx-auto mb-2 bg-gradient-wellness rounded-lg flex items-center justify-center">
                  <TrendingUp className="w-6 h-6 text-white" />
                </div>
                <p className="text-sm font-medium text-foreground mb-1">Analytics</p>
                <p className="text-xs text-muted-foreground">View detailed stats</p>
              </CardContent>
            </Card>
            
            <Card 
              className="cursor-pointer hover:shadow-medium transition-all duration-300 transform hover:scale-105 border-border/50"
              onClick={() => navigate('/DailyJournal')}
            >
              <CardContent className="p-4 text-center">
                <div className="w-12 h-12 mx-auto mb-2 bg-yellow-500 rounded-lg flex items-center justify-center">
                  <Heart className="w-6 h-6 text-white" />
                </div>
                <p className="text-sm font-medium text-foreground mb-1">Quick Mood</p>
                <p className="text-xs text-muted-foreground">Log feeling now</p>
              </CardContent>
            </Card>
            
            <Card 
              className="cursor-pointer hover:shadow-medium transition-all duration-300 transform hover:scale-105 border-border/50"
              onClick={() => navigate('/habits')}
            >
              <CardContent className="p-4 text-center">
                <div className="w-12 h-12 mx-auto mb-2 bg-green-500 rounded-lg flex items-center justify-center">
                  <CheckCircle2 className="w-6 h-6 text-white" />
                </div>
                <p className="text-sm font-medium text-foreground mb-1">Quick Check</p>
                <p className="text-xs text-muted-foreground">Mark habits done</p>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </div>
  );
}