// src/components/OrganizationDashboard/InviteMembers.tsx
import { useState } from 'react';
import { supabase } from '../../../supabaseClient';
import { Organization } from '../../../types/organization';
import { Button } from '../../ui/Button';
import { Input } from '../../ui/Input';
import { Select } from '../../ui/Select';

interface InviteMembersModalProps {
  organization: Organization;
  onClose: () => void;
}

export function InviteMembersModal({ organization, onClose }: InviteMembersModalProps) {
  const [emails, setEmails] = useState('');
  const [role, setRole] = useState<'admin' | 'editor' | 'viewer'>('editor');
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [success, setSuccess] = useState<string | null>(null);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    setError(null);
    setSuccess(null);

    try {
      const emailList = emails.split(',').map(email => email.trim());
      
      // Send invites
      const { error: inviteError } = await supabase
        .from('organization_invites')
        .insert(emailList.map(email => ({
          organization_id: organization.id,
          email,
          role,
          invited_at: new Date().toISOString()
        })));

      if (inviteError) throw inviteError;

      setSuccess(`${emailList.length} invites sent successfully!`);
      setEmails('');
    } catch (err) {
      setError(err.message || 'Failed to send invites');
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="modal-overlay">
      <div className="invite-modal">
        <h2>Invite Members</h2>
        <button className="close-btn" onClick={onClose}>×</button>
        
        <form onSubmit={handleSubmit}>
          <div className="form-group">
            <label>Email Addresses</label>
            <Input
              as="textarea"
              value={emails}
              onChange={(e) => setEmails(e.target.value)}
              placeholder="Enter comma-separated email addresses"
              required
            />
            <p className="hint">Separate multiple emails with commas</p>
          </div>
          
          <div className="form-group">
            <label>Role</label>
            <Select
              value={role}
              onChange={(value) => setRole(value as any)}
              options={[
                { value: 'admin', label: 'Admin' },
                { value: 'editor', label: 'Editor' },
                { value: 'viewer', label: 'Viewer' }
              ]}
            />
          </div>
          
          {error && <div className="error-message">{error}</div>}
          {success && <div className="success-message">{success}</div>}
          
          <div className="modal-actions">
            <Button type="button" variant="secondary" onClick={onClose}>
              Cancel
            </Button>
            <Button type="submit" loading={loading}>
              Send Invites
            </Button>
          </div>
        </form>
      </div>
    </div>
  );
}