import React, { useEffect, useState } from "react";
import { supabase } from "@/components/supabaseClient";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { useUser } from "@supabase/auth-helpers-react";
import { BarChart, Bar, XAxis, YAxis, Tooltip, ResponsiveContainer } from "recharts";
import { 
  Select, 
  SelectContent, 
  SelectItem, 
  SelectTrigger, 
  SelectValue 
} from "@/components/ui/select";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Mail, UserCheck, Clock, XCircle, CheckCircle } from "lucide-react";

const OrganizationDashboard = () => {
  const user = useUser();
  const [orgUsers, setOrgUsers] = useState([]);
  const [emailToInvite, setEmailToInvite] = useState("");
  const [role, setRole] = useState("member");
  const [insightsData, setInsightsData] = useState([]);
  const [orgRole, setOrgRole] = useState(null);
  const [currentOrg, setCurrentOrg] = useState(null);
  const [invites, setInvites] = useState([]);
  const [isLoading, setIsLoading] = useState(false);

  useEffect(() => {
    if (user) {
      fetchOrgData();
      fetchOrgUsers();
      fetchInsights();
      fetchInvites();
    }
  }, [user]);

  const fetchOrgData = async () => {
    // Get user's organization and role
    const { data: memberData, error: memberError } = await supabase
      .from("organization_members")
      .select("role, organizations(*)")
      .eq("user_id", user?.id)
      .single();

    if (!memberError && memberData) {
      setOrgRole(memberData.role);
      setCurrentOrg(memberData.organizations);
    }
  };

  const fetchOrgUsers = async () => {
    const { data, error } = await supabase
      .from("organization_members")
      .select("*, profiles(full_name, email)");
    if (!error && data) setOrgUsers(data);
  };

  const fetchInvites = async () => {
    const { data, error } = await supabase
      .from("organization_invites")
      .select("*")
      .order("created_at", { ascending: false });
    if (!error && data) setInvites(data);
  };

  const fetchInsights = async () => {
    const { data, error } = await supabase
      .from("organization_insights")
      .select("month, value");
    if (!error && data) setInsightsData(data);
  };

const handleInvite = async () => {
  if (!emailToInvite || !role) {
    alert("Please enter an email and select a role");
    return;
  }

  setIsLoading(true);
  try {
    const email = emailToInvite.toLowerCase().trim();

    // 1. Check if user already exists
    const { data: existingUser } = await supabase
      .from('profiles')
      .select('id, email')
      .eq('email', email)
      .maybeSingle();

    if (existingUser) {
      // User exists - add directly to organization
      const { error: memberError } = await supabase
        .from('organization_members')
        .insert({
          user_id: existingUser.id,
          organization_id: currentOrg?.id,
          role: role,
        });

      if (memberError) {
        if (memberError.code === '23505') {
          alert("User is already a member of this organization");
        } else {
          throw new Error(`Failed to add user: ${memberError.message}`);
        }
      } else {
        alert("✅ User added to organization successfully!");
      }
    } else {
      // 2. User doesn't exist - create custom invite
      const inviteToken = generateInviteToken();
      
      const { data: invite, error: inviteError } = await supabase
        .from('organization_invites')
        .insert({
          email: email,
          role: role,
          organization_id: currentOrg?.id,
          invited_by: user?.id,
          status: 'pending',
          token: inviteToken,
          expires_at: new Date(Date.now() + 7 * 24 * 60 * 60 * 1000).toISOString(),
        })
        .select()
        .single();

      if (inviteError) throw inviteError;

      // 3. Create magic link
      const magicLink = `${window.location.origin}/auth?invite=${inviteToken}&email=${encodeURIComponent(email)}&org=${currentOrg?.id}&role=${role}`;
      
      // 4. Store for reference
      await supabase
        .from('invite_links')
        .insert({
          invite_id: invite.id,
          url: magicLink,
          created_at: new Date().toISOString(),
        });

      // 5. Copy to clipboard and show to admin
      try {
        await navigator.clipboard.writeText(magicLink);
        alert(`✅ Invite created! Magic link copied to clipboard:\n${magicLink}`);
      } catch {
        alert(`✅ Invite created! Share this link:\n${magicLink}`);
      }
    }

    setEmailToInvite("");
    setRole("member");
    fetchInvites();

  } catch (error: any) {
    console.error("Invite Error:", error);
    alert(`❌ Error: ${error.message}`);
  } finally {
    setIsLoading(false);
  }
};

// Generate secure token
const generateInviteToken = () => {
  const array = new Uint8Array(20);
  window.crypto.getRandomValues(array);
  return Array.from(array, byte => byte.toString(16).padStart(2, '0')).join('');
};

// Helper function to generate invite token
const generateInviteToken = () => {
  return Math.random().toString(36).substring(2) + Date.now().toString(36);
};

const handleCustomInviteFallback = async () => {
  try {
    // Create invite record without relying on Supabase Auth
    const { data: invite, error: inviteError } = await supabase
      .from('organization_invites')
      .insert({
        email: emailToInvite,
        role: role,
        organization_id: currentOrg?.id,
        invited_by: user?.id,
        status: 'pending',
        expires_at: new Date(Date.now() + 7 * 24 * 60 * 60 * 1000).toISOString(),
      })
      .select()
      .single();

    if (inviteError) throw inviteError;

    // Use Edge Function to send custom email
    const { error: emailError } = await supabase.functions.invoke('send-custom-invite', {
      body: {
        email: emailToInvite,
        role: role,
        organization_name: currentOrg?.name,
        invite_id: invite.id,
        invite_url: `${window.location.origin}/signup?invite=${invite.id}&email=${encodeURIComponent(emailToInvite)}`,
      },
    });

    if (emailError) {
      console.warn('Email sending failed, but invite was created:', emailError);
    }

    alert("Invite created successfully! User can sign up and join your organization.");
    
  } catch (error: any) {
    console.error("Custom invite fallback error:", error);
    throw error;
  }
};

  const getStatusBadge = (status) => {
    const statusConfig = {
      pending: { variant: "secondary", icon: Clock, text: "Pending" },
      accepted: { variant: "default", icon: CheckCircle, text: "Accepted" },
      expired: { variant: "destructive", icon: XCircle, text: "Expired" },
      revoked: { variant: "destructive", icon: XCircle, text: "Revoked" },
    };
    
    const config = statusConfig[status] || statusConfig.pending;
    const IconComponent = config.icon;
    
    return (
      <Badge variant={config.variant} className="flex items-center gap-1">
        <IconComponent className="w-3 h-3" />
        {config.text}
      </Badge>
    );
  };

  return (
    <div className="p-6 max-w-6xl mx-auto">
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-3xl font-bold">Organization Dashboard</h1>
        {currentOrg && (
          <Badge variant="outline" className="text-lg px-4 py-2">
            {currentOrg.name}
          </Badge>
        )}
      </div>

      <Tabs defaultValue="members" className="w-full">
        <TabsList className="grid grid-cols-5 mb-6">
          <TabsTrigger value="members">Manage Users</TabsTrigger>
          <TabsTrigger value="trackers">Trackers</TabsTrigger>
          <TabsTrigger value="dashboard">Custom Dashboards</TabsTrigger>
          <TabsTrigger value="updates">Updates</TabsTrigger>
          <TabsTrigger value="insights">Insights</TabsTrigger>
        </TabsList>

        <TabsContent value="members">
          {orgRole === "admin" && (
            <Card className="mb-6">
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Mail className="w-5 h-5" />
                  Invite New Member
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="flex gap-4 items-end">
                  <div className="flex-1">
                    <label className="text-sm font-medium mb-2 block">Email Address</label>
                    <Input
                      type="email"
                      placeholder="Enter email to invite"
                      value={emailToInvite}
                      onChange={(e) => setEmailToInvite(e.target.value)}
                      className="w-full"
                    />
                  </div>
                  <div className="w-32">
                    <label className="text-sm font-medium mb-2 block">Role</label>
                    <Select value={role} onValueChange={setRole}>
                      <SelectTrigger>
                        <SelectValue placeholder="Role" />
                      </SelectTrigger>
                      <SelectContent>
                        <SelectItem value="member">Member</SelectItem>
                        <SelectItem value="admin">Admin</SelectItem>
                      </SelectContent>
                    </Select>
                  </div>
                  <Button 
                    onClick={handleInvite} 
                    disabled={isLoading || !emailToInvite}
                    className="whitespace-nowrap"
                  >
                    {isLoading ? "Sending..." : "Send Invite"}
                  </Button>
                </div>
              </CardContent>
            </Card>
          )}

          <div className="grid gap-6 md:grid-cols-2">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <UserCheck className="w-5 h-5" />
                  Organization Members ({orgUsers.length})
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  {orgUsers.map((user) => (
                    <div key={user.id} className="flex justify-between items-center p-3 border rounded-lg">
                      <div>
                        <p className="font-medium">{user.profiles?.full_name || user.profiles?.email}</p>
                        <p className="text-sm text-muted-foreground">{user.profiles?.email}</p>
                      </div>
                      <Badge variant={user.role === "admin" ? "default" : "secondary"}>
                        {user.role}
                      </Badge>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Mail className="w-5 h-5" />
                  Pending Invites ({invites.filter(i => i.status === 'pending').length})
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  {invites.map((invite) => (
                    <div key={invite.id} className="flex justify-between items-center p-3 border rounded-lg">
                      <div>
                        <p className="font-medium">{invite.email}</p>
                        <p className="text-sm text-muted-foreground">Role: {invite.role}</p>
                      </div>
                      {getStatusBadge(invite.status)}
                    </div>
                  ))}
                  {invites.length === 0 && (
                    <p className="text-muted-foreground text-center py-4">No pending invites</p>
                  )}
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>

        {/* Other tabs remain the same */}
        <TabsContent value="trackers">
          <h2 className="text-lg font-semibold mb-2">Assign Trackers</h2>
          <p className="text-sm text-muted-foreground">[Tracker management UI goes here]</p>
        </TabsContent>

        <TabsContent value="dashboard">
          <h2 className="text-lg font-semibold mb-2">Customize Dashboards</h2>
          <p className="text-sm text-muted-foreground">[Widget configuration or layout options go here]</p>
        </TabsContent>

        <TabsContent value="updates">
          <h2 className="text-lg font-semibold mb-2">Input / Review Updates</h2>
          <p className="text-sm text-muted-foreground">[Organization updates or logs section]</p>
        </TabsContent>

        <TabsContent value="insights">
          <h2 className="text-lg font-semibold mb-2">Org-Wide Reports</h2>
          <ResponsiveContainer width="100%" height={300}>
            <BarChart data={insightsData}>
              <XAxis dataKey="month" />
              <YAxis />
              <Tooltip />
              <Bar dataKey="value" fill="#8884d8" />
            </BarChart>
          </ResponsiveContainer>
        </TabsContent>
      </Tabs>
    </div>
  );
};

export default OrganizationDashboard;