
import { useState, useEffect } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { Badge } from "@/components/ui/badge";
import { Calendar, TrendingUp, TrendingDown, Plus, Minus, FileText } from "lucide-react";
import { format, startOfMonth, endOfMonth } from "date-fns";

export const MonthlyReview = () => {
  const [currentMonth] = useState(new Date());
  const [achievements, setAchievements] = useState("");
  const [workedWell, setWorkedWell] = useState("");
  const [didntWork, setDidntWork] = useState("");
  const [doMore, setDoMore] = useState("");
  const [doLess, setDoLess] = useState("");
  const [pastReviews, setPastReviews] = useState<any[]>([]);

  useEffect(() => {
    // Load past reviews from localStorage
    const saved = JSON.parse(localStorage.getItem('monthlyReviews') || '[]');
    setPastReviews(saved);
  }, []);

  const handleSave = () => {
    const review = {
      month: format(currentMonth, 'MMMM yyyy'),
      date: new Date().toISOString(),
      achievements,
      workedWell,
      didntWork,
      doMore,
      doLess
    };

    const existingReviews = JSON.parse(localStorage.getItem('monthlyReviews') || '[]');
    existingReviews.push(review);
    localStorage.setItem('monthlyReviews', JSON.stringify(existingReviews));
    setPastReviews(existingReviews);

    // Reset form
    setAchievements("");
    setWorkedWell("");
    setDidntWork("");
    setDoMore("");
    setDoLess("");

    alert("Monthly review saved successfully!");
  };

  const getJournalStats = () => {
    const entries = JSON.parse(localStorage.getItem('journalEntries') || '[]');
    const monthStart = startOfMonth(currentMonth);
    const monthEnd = endOfMonth(currentMonth);
    
    const monthEntries = entries.filter((entry: any) => {
      const entryDate = new Date(entry.date);
      return entryDate >= monthStart && entryDate <= monthEnd;
    });

    return {
      totalEntries: monthEntries.length,
      commonMood: 'happy', // Simplified for demo
      gratitudeCount: monthEntries.reduce((acc: number, entry: any) => acc + (entry.gratitude?.length || 0), 0)
    };
  };

  const stats = getJournalStats();

  return (
    <div className="space-y-6">
      {/* Header */}
      <Card className="bg-gradient-to-r from-green-500 to-teal-500 text-white border-0">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Calendar className="h-6 w-6" />
            Monthly Review - {format(currentMonth, 'MMMM yyyy')}
          </CardTitle>
        </CardHeader>
      </Card>

      {/* Month Statistics */}
      <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
        <CardHeader>
          <CardTitle>This Month's Journey</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="text-center p-4 bg-blue-50 rounded-lg">
              <div className="text-2xl font-bold text-blue-600">{stats.totalEntries}</div>
              <div className="text-sm text-blue-800">Journal Entries</div>
            </div>
            <div className="text-center p-4 bg-yellow-50 rounded-lg">
              <div className="text-2xl font-bold text-yellow-600">{stats.gratitudeCount}</div>
              <div className="text-sm text-yellow-800">Gratitude Items</div>
            </div>
            <div className="text-center p-4 bg-green-50 rounded-lg">
              <div className="text-2xl font-bold text-green-600">😊</div>
              <div className="text-sm text-green-800">Most Common Mood</div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Review Questions */}
      <div className="space-y-6">
        <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
          <CardHeader>
            <CardTitle className="flex items-center gap-2 text-green-600">
              <TrendingUp className="h-5 w-5" />
              What I Achieved This Month
            </CardTitle>
          </CardHeader>
          <CardContent>
            <Textarea
              placeholder="List your accomplishments, milestones, and victories from this month..."
              value={achievements}
              onChange={(e) => setAchievements(e.target.value)}
              className="min-h-32"
            />
          </CardContent>
        </Card>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
            <CardHeader>
              <CardTitle className="flex items-center gap-2 text-blue-600">
                <TrendingUp className="h-5 w-5" />
                What Worked Well
              </CardTitle>
            </CardHeader>
            <CardContent>
              <Textarea
                placeholder="What strategies, habits, or approaches were successful?"
                value={workedWell}
                onChange={(e) => setWorkedWell(e.target.value)}
                className="min-h-24"
              />
            </CardContent>
          </Card>

          <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
            <CardHeader>
              <CardTitle className="flex items-center gap-2 text-red-600">
                <TrendingDown className="h-5 w-5" />
                What Didn't Work
              </CardTitle>
            </CardHeader>
            <CardContent>
              <Textarea
                placeholder="What challenges or obstacles did you encounter? What would you change?"
                value={didntWork}
                onChange={(e) => setDidntWork(e.target.value)}
                className="min-h-24"
              />
            </CardContent>
          </Card>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
            <CardHeader>
              <CardTitle className="flex items-center gap-2 text-green-600">
                <Plus className="h-5 w-5" />
                Do More Of
              </CardTitle>
            </CardHeader>
            <CardContent>
              <Textarea
                placeholder="What activities, habits, or behaviors do you want to increase?"
                value={doMore}
                onChange={(e) => setDoMore(e.target.value)}
                className="min-h-24"
              />
            </CardContent>
          </Card>

          <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
            <CardHeader>
              <CardTitle className="flex items-center gap-2 text-orange-600">
                <Minus className="h-5 w-5" />
                Do Less Of
              </CardTitle>
            </CardHeader>
            <CardContent>
              <Textarea
                placeholder="What would you like to reduce or eliminate from your routine?"
                value={doLess}
                onChange={(e) => setDoLess(e.target.value)}
                className="min-h-24"
              />
            </CardContent>
          </Card>
        </div>
      </div>

      {/* Save Button */}
      <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
        <CardContent className="pt-6">
          <Button 
            onClick={handleSave}
            className="w-full bg-gradient-to-r from-green-500 to-teal-600 hover:from-green-600 hover:to-teal-700 text-white font-medium py-3"
          >
            Save Monthly Review
          </Button>
        </CardContent>
      </Card>

      {/* Past Reviews */}
      {pastReviews.length > 0 && (
        <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <FileText className="h-5 w-5" />
              Past Reviews
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {pastReviews.slice(-3).reverse().map((review, index) => (
                <div key={index} className="p-4 bg-gray-50 rounded-lg">
                  <div className="flex justify-between items-center mb-2">
                    <Badge variant="secondary">{review.month}</Badge>
                    <span className="text-sm text-gray-500">
                      {format(new Date(review.date), 'MMM d')}
                    </span>
                  </div>
                  <p className="text-sm text-gray-600 line-clamp-2">
                    {review.achievements?.substring(0, 100)}...
                  </p>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>
      )}
    </div>
  );
};
