import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { ArrowLeft, Lightbulb, Plus, Search, Filter, Star, Clock, Users, TrendingUp, CheckCircle, AlertCircle, Zap, Edit, Trash2, MoreVertical } from 'lucide-react';
import { useNavigate } from "react-router-dom";
import { supabase } from '@/components/supabaseClient';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";

interface InnovationTrackerProps {
  onBack: () => void;
}

interface Innovation {
  id: string;
  title: string;
  description: string;
  category: string;
  status: string;
  priority: string;
  submittedDate: string;
  implementationDate?: string;
  impact: string;
  votes: number;
  author: string;
  tags: string[];
}

const categories = ['All', 'Technology', 'Product', 'AI/ML', 'Sustainability', 'Process'];
const statuses = ['All', 'brainstorming', 'in-review', 'approved', 'implemented'];

const getStatusColor = (status: string) => {
  switch (status) {
    case 'approved': return 'bg-green-100 text-green-800 border-green-200';
    case 'in-review': return 'bg-yellow-100 text-yellow-800 border-yellow-200';
    case 'implemented': return 'bg-blue-100 text-blue-800 border-blue-200';
    default: return 'bg-gray-100 text-gray-800 border-gray-200';
  }
};

const getPriorityIcon = (priority: string) => {
  switch (priority) {
    case 'high': return <AlertCircle className="w-4 h-4 text-red-500" />;
    case 'medium': return <Clock className="w-4 h-4 text-yellow-500" />;
    default: return <CheckCircle className="w-4 h-4 text-green-500" />;
  }
};

export function InnovationTracker({ onBack }: InnovationTrackerProps) {
  const navigate = useNavigate();
  const [selectedCategory, setSelectedCategory] = useState('All');
  const [selectedStatus, setSelectedStatus] = useState('All');
  const [searchTerm, setSearchTerm] = useState('');
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false);
  const [innovations, setInnovations] = useState<Innovation[]>([]);
  const [loading, setLoading] = useState(true);
  const [formData, setFormData] = useState({
    title: '',
    description: '',
    category: 'Technology',
    impact: 'High'
  });
  const [editingInnovation, setEditingInnovation] = useState<Innovation | null>(null);
  const [editFormData, setEditFormData] = useState({
    title: '',
    description: '',
    category: 'Technology',
    impact: 'High',
    status: 'brainstorming'
  });

  // Fetch innovations from Supabase
  useEffect(() => {
    fetchInnovations();
  }, []);

  const fetchInnovations = async () => {
    try {
      setLoading(true);
      const { data, error } = await supabase
        .from('innovations')
        .select('*')
        .order('submittedDate', { ascending: false });

      if (error) {
        throw error;
      }

      if (data) {
        setInnovations(data);
      }
    } catch (error) {
      console.error('Error fetching innovations:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleSubmitIdea = async () => {
    try {
      const newInnovation = {
        title: formData.title,
        description: formData.description,
        category: formData.category,
        status: 'brainstorming',
        priority: formData.impact === 'High' ? 'high' : formData.impact === 'Medium' ? 'medium' : 'low',
        submittedDate: new Date().toISOString().split('T')[0],
        impact: formData.impact,
        votes: 0,
        author: 'You',
        tags: []
      };

      const { data, error } = await supabase
        .from('innovations')
        .insert([newInnovation])
        .select();

      if (error) {
        throw error;
      }

      if (data) {
        await fetchInnovations();
        setIsDialogOpen(false);
        setFormData({
          title: '',
          description: '',
          category: 'Technology',
          impact: 'High'
        });
      }
    } catch (error) {
      console.error('Error submitting idea:', error);
    }
  };

  const handleEditIdea = async () => {
    if (!editingInnovation) return;

    try {
      const { error } = await supabase
        .from('innovations')
        .update({
          title: editFormData.title,
          description: editFormData.description,
          category: editFormData.category,
          impact: editFormData.impact,
          status: editFormData.status,
          priority: editFormData.impact === 'High' ? 'high' : editFormData.impact === 'Medium' ? 'medium' : 'low'
        })
        .eq('id', editingInnovation.id);

      if (error) {
        throw error;
      }

      await fetchInnovations();
      setIsEditDialogOpen(false);
      setEditingInnovation(null);
    } catch (error) {
      console.error('Error updating idea:', error);
    }
  };

  const handleDeleteIdea = async (id: string) => {
    if (!confirm('Are you sure you want to delete this innovation idea?')) {
      return;
    }

    try {
      const { error } = await supabase
        .from('innovations')
        .delete()
        .eq('id', id);

      if (error) {
        throw error;
      }

      await fetchInnovations();
    } catch (error) {
      console.error('Error deleting idea:', error);
    }
  };

  const openEditDialog = (innovation: Innovation) => {
    setEditingInnovation(innovation);
    setEditFormData({
      title: innovation.title,
      description: innovation.description,
      category: innovation.category,
      impact: innovation.impact,
      status: innovation.status
    });
    setIsEditDialogOpen(true);
  };

  const filteredIdeas = innovations.filter(idea => {
    const matchesCategory = selectedCategory === 'All' || idea.category === selectedCategory;
    const matchesStatus = selectedStatus === 'All' || idea.status === selectedStatus;
    const matchesSearch = idea.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         idea.description.toLowerCase().includes(searchTerm.toLowerCase());
    return matchesCategory && matchesStatus && matchesSearch;
  });

  const stats = [
    {
      title: 'Total Ideas',
      value: innovations.length.toString(),
      description: 'Submitted this quarter',
      icon: Lightbulb,
      color: 'text-primary'
    },
    {
      title: 'Approved',
      value: innovations.filter(i => i.status === 'approved').length.toString(),
      description: 'Ready for implementation',
      icon: CheckCircle,
      color: 'text-green-500'
    },
    {
      title: 'High Impact',
      value: innovations.filter(i => i.impact === 'High').length.toString(),
      description: 'Game-changing ideas',
      icon: Zap,
      color: 'text-amber-500'
    }
  ];

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="border-b bg-card/50 backdrop-blur-sm sticky top-0 z-50">
        <div className="container mx-auto px-6 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-4">
              <Button variant="ghost" size="sm"  onClick={() => navigate("/CareerDevelopment")}
                className="w-4 h-4 mr-2">
                <ArrowLeft className="w-4 h-4" />
                </Button>
              <div className="w-10 h-10 rounded-lg bg-gradient-growth flex items-center justify-center">
                <Lightbulb className="w-6 h-6 text-white" />
              </div>
              <div>
                <h1 className="text-2xl font-bold bg-gradient-hero bg-clip-text text-transparent">
                  Innovation Tracker
                </h1>
                <p className="text-sm text-muted-foreground">Manage ideas and track innovation projects</p>
              </div>
            </div>
            <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
              <DialogTrigger asChild>
                <Button className="bg-gradient-primary text-white hover:opacity-90">
                  <Plus className="w-4 h-4 mr-2" />
                  Submit Idea
                </Button>
              </DialogTrigger>
              <DialogContent className="max-w-2xl">
                <DialogHeader>
                  <DialogTitle>Submit New Innovation Idea</DialogTitle>
                  <DialogDescription>
                    Share your innovative ideas to help improve our organization
                  </DialogDescription>
                </DialogHeader>
                <div className="space-y-4">
                  <div>
                    <label className="text-sm font-medium text-foreground">Title</label>
                    <Input 
                      placeholder="Brief, descriptive title for your idea" 
                      className="mt-1" 
                      value={formData.title}
                      onChange={(e) => setFormData({...formData, title: e.target.value})}
                    />
                  </div>
                  <div>
                    <label className="text-sm font-medium text-foreground">Description</label>
                    <Textarea 
                      placeholder="Detailed description of your innovation idea" 
                      className="mt-1 min-h-[100px]"
                      value={formData.description}
                      onChange={(e) => setFormData({...formData, description: e.target.value})}
                    />
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <label className="text-sm font-medium text-foreground">Category</label>
                      <select 
                        className="w-full mt-1 rounded-md border border-input bg-background px-3 py-2 text-sm"
                        value={formData.category}
                        onChange={(e) => setFormData({...formData, category: e.target.value})}
                      >
                        <option value="Technology">Technology</option>
                        <option value="Product">Product</option>
                        <option value="AI/ML">AI/ML</option>
                        <option value="Sustainability">Sustainability</option>
                        <option value="Process">Process</option>
                      </select>
                    </div>
                    <div>
                      <label className="text-sm font-medium text-foreground">Expected Impact</label>
                      <select 
                        className="w-full mt-1 rounded-md border border-input bg-background px-3 py-2 text-sm"
                        value={formData.impact}
                        onChange={(e) => setFormData({...formData, impact: e.target.value})}
                      >
                        <option value="High">High</option>
                        <option value="Medium">Medium</option>
                        <option value="Low">Low</option>
                      </select>
                    </div>
                  </div>
                  <div className="flex justify-end space-x-2 pt-4">
                    <Button variant="outline" onClick={() => setIsDialogOpen(false)}>
                      Cancel
                    </Button>
                    <Button 
                      className="bg-gradient-primary text-white hover:opacity-90"
                      onClick={handleSubmitIdea}
                      disabled={!formData.title || !formData.description}
                    >
                      Submit Idea
                    </Button>
                  </div>
                </div>
              </DialogContent>
            </Dialog>

            {/* Edit Dialog */}
            <Dialog open={isEditDialogOpen} onOpenChange={setIsEditDialogOpen}>
              <DialogContent className="max-w-2xl">
                <DialogHeader>
                  <DialogTitle>Edit Innovation Idea</DialogTitle>
                  <DialogDescription>
                    Update your innovation idea
                  </DialogDescription>
                </DialogHeader>
                <div className="space-y-4">
                  <div>
                    <label className="text-sm font-medium text-foreground">Title</label>
                    <Input 
                      placeholder="Brief, descriptive title for your idea" 
                      className="mt-1" 
                      value={editFormData.title}
                      onChange={(e) => setEditFormData({...editFormData, title: e.target.value})}
                    />
                  </div>
                  <div>
                    <label className="text-sm font-medium text-foreground">Description</label>
                    <Textarea 
                      placeholder="Detailed description of your innovation idea" 
                      className="mt-1 min-h-[100px]"
                      value={editFormData.description}
                      onChange={(e) => setEditFormData({...editFormData, description: e.target.value})}
                    />
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <label className="text-sm font-medium text-foreground">Category</label>
                      <select 
                        className="w-full mt-1 rounded-md border border-input bg-background px-3 py-2 text-sm"
                        value={editFormData.category}
                        onChange={(e) => setEditFormData({...editFormData, category: e.target.value})}
                      >
                        <option value="Technology">Technology</option>
                        <option value="Product">Product</option>
                        <option value="AI/ML">AI/ML</option>
                        <option value="Sustainability">Sustainability</option>
                        <option value="Process">Process</option>
                      </select>
                    </div>
                    <div>
                      <label className="text-sm font-medium text-foreground">Expected Impact</label>
                      <select 
                        className="w-full mt-1 rounded-md border border-input bg-background px-3 py-2 text-sm"
                        value={editFormData.impact}
                        onChange={(e) => setEditFormData({...editFormData, impact: e.target.value})}
                      >
                        <option value="High">High</option>
                        <option value="Medium">Medium</option>
                        <option value="Low">Low</option>
                      </select>
                    </div>
                  </div>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <label className="text-sm font-medium text-foreground">Status</label>
                      <select 
                        className="w-full mt-1 rounded-md border border-input bg-background px-3 py-2 text-sm"
                        value={editFormData.status}
                        onChange={(e) => setEditFormData({...editFormData, status: e.target.value})}
                      >
                        <option value="brainstorming">Brainstorming</option>
                        <option value="in-review">In Review</option>
                        <option value="approved">Approved</option>
                        <option value="implemented">Implemented</option>
                      </select>
                    </div>
                  </div>
                  <div className="flex justify-end space-x-2 pt-4">
                    <Button variant="outline" onClick={() => setIsEditDialogOpen(false)}>
                      Cancel
                    </Button>
                    <Button 
                      className="bg-gradient-primary text-white hover:opacity-90"
                      onClick={handleEditIdea}
                      disabled={!editFormData.title || !editFormData.description}
                    >
                      Update Idea
                    </Button>
                  </div>
                </div>
              </DialogContent>
            </Dialog>
          </div>
        </div>
      </div>

      {/* Main Content */}
      <div className="container mx-auto px-6 py-8">
        {/* Stats */}
        <div className="mb-8 animate-fade-in">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            {stats.map((stat, index) => (
              <Card key={stat.title} className="border-border/50">
                <CardHeader className="pb-3">
                  <CardTitle className="text-base text-foreground flex items-center">
                    <stat.icon className={`w-5 h-5 mr-2 ${stat.color}`} />
                    {stat.title}
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="text-2xl font-bold text-foreground">{stat.value}</div>
                  <p className="text-sm text-muted-foreground">{stat.description}</p>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>

        {/* Filters */}
        <div className="mb-6 animate-slide-up" style={{ animationDelay: '0.2s' }}>
          <div className="flex flex-col md:flex-row gap-4 items-center justify-between">
            <div className="flex flex-wrap gap-2">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
                <Input
                  placeholder="Search ideas..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="pl-10 w-64"
                />
              </div>
            </div>
            <div className="flex gap-2">
              <select 
                value={selectedCategory}
                onChange={(e) => setSelectedCategory(e.target.value)}
                className="rounded-md border border-input bg-background px-3 py-2 text-sm"
              >
                {categories.map(cat => <option key={cat} value={cat}>{cat}</option>)}
              </select>
              <select 
                value={selectedStatus}
                onChange={(e) => setSelectedStatus(e.target.value)}
                className="rounded-md border border-input bg-background px-3 py-2 text-sm"
              >
                {statuses.map(status => <option key={status} value={status}>{status}</option>)}
              </select>
            </div>
          </div>
        </div>

        {/* Ideas Grid */}
        <div className="animate-slide-up" style={{ animationDelay: '0.4s' }}>
          {loading ? (
            <div className="text-center py-8">
              <p>Loading innovations...</p>
            </div>
          ) : filteredIdeas.length === 0 ? (
            <div className="text-center py-8">
              <p>No innovations found. Be the first to submit an idea!</p>
            </div>
          ) : (
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {filteredIdeas.map((idea) => (
                <Card key={idea.id} className="border-border/50 hover:shadow-strong transition-all duration-300">
                  <CardHeader className="pb-3">
                    <div className="flex items-start justify-between">
                      <div className="flex-1">
                        <div className="flex items-center space-x-2 mb-2">
                          {getPriorityIcon(idea.priority)}
                          <CardTitle className="text-lg text-foreground">{idea.title}</CardTitle>
                        </div>
                        <div className="flex items-center space-x-2">
                          <Badge variant="outline" className="text-xs">
                            {idea.category}
                          </Badge>
                          <Badge className={`text-xs ${getStatusColor(idea.status)}`}>
                            {idea.status}
                          </Badge>
                        </div>
                      </div>
                      <DropdownMenu>
                        <DropdownMenuTrigger asChild>
                          <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                            <MoreVertical className="h-4 w-4" />
                          </Button>
                        </DropdownMenuTrigger>
                        <DropdownMenuContent align="end">
                          <DropdownMenuItem onClick={() => openEditDialog(idea)}>
                            <Edit className="w-4 h-4 mr-2" />
                            Edit
                          </DropdownMenuItem>
                          <DropdownMenuItem 
                            onClick={() => handleDeleteIdea(idea.id)}
                            className="text-red-600"
                          >
                            <Trash2 className="w-4 h-4 mr-2" />
                            Delete
                          </DropdownMenuItem>
                        </DropdownMenuContent>
                      </DropdownMenu>
                    </div>
                  </CardHeader>
                  <CardContent>
                    <CardDescription className="text-muted-foreground mb-4">
                      {idea.description}
                    </CardDescription>
                    
                    <div className="flex flex-wrap gap-1 mb-4">
                      {idea.tags?.map(tag => (
                        <Badge key={tag} variant="secondary" className="text-xs">
                          {tag}
                        </Badge>
                      ))}
                    </div>

                    <div className="flex items-center justify-between text-sm text-muted-foreground">
                      <div className="flex items-center space-x-4">
                        <div className="flex items-center space-x-1">
                          <Star className="w-4 h-4" />
                          <span>{idea.votes}</span>
                        </div>
                        <div className="flex items-center space-x-1">
                          <Users className="w-4 h-4" />
                          <span>{idea.author}</span>
                        </div>
                      </div>
                      <div className="flex items-center space-x-1">
                        <Clock className="w-3 h-3" />
                        <span>{idea.submittedDate}</span>
                      </div>
                    </div>
                    
                    {idea.implementationDate && (
                      <div className="mt-2 p-2 bg-green-50 dark:bg-green-900/20 rounded-md">
                        <p className="text-xs text-green-700 dark:text-green-300">
                          Implementation scheduled: {idea.implementationDate}
                        </p>
                      </div>
                    )}
                  </CardContent>
                </Card>
              ))}
            </div>
          )}
        </div>
      </div>
    </div>
  );
}