import { useState, useEffect } from "react";
import { supabase } from '@/components/supabaseClient';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Textarea } from '@/components/ui/textarea';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';

import { useNavigate } from 'react-router-dom';

export default function Goal() {
  const navigate = useNavigate();

  const [title, setTitle] = useState('');
  const [description, setDescription] = useState('');
  const [targetDate, setTargetDate] = useState('');
  const [goals, setGoals] = useState([]);
  const [loading, setLoading] = useState(false);
  const [editId, setEditId] = useState(null);
  const [completion_percentage, setCompletionPercentage] = useState(0);

  useEffect(() => {
     const fetchGoals = async () => {
    const { data: { user } } = await supabase.auth.getUser();
    if (!user) return;

    const { data, error } = await supabase
      .from('goals')
      .select('*')
      .eq('user_id', user.id)
      .order('created_at', { ascending: false });

    if (error) {
      console.error('Error fetching goals:', error);
      return;
    }

    if (data) {
      // Filter out invalid goal status-percentage combinations
      const validGoals = data.filter(goal => {
        return (goal.status === 'completed' && goal.completion_percentage === 100) ||
               (goal.status === 'pending' && goal.completion_percentage === 0);
      });
    }
  };

  }, []);

  const fetchGoals = async () => {
    const { data: { user } } = await supabase.auth.getUser();
    const { data, error } = await supabase
      .from('goals')
      .select('*')
      .eq('user_id', user.id);

    if (!error) setGoals(data);
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);
    const { data: { user } } = await supabase.auth.getUser();

    if (editId) {
      await supabase
        .from('goals')
        .update({ title, description, target_date: targetDate, status: 'Pending', completion_percentage })
        .eq('id', editId);
    } else {
      await supabase.from('goals').insert([
        {
          user_id: user.id,
          title,
          description,
          target_date: targetDate || null,
          status: 'Pending',
        },
      ]);
    }

    setTitle('');
    setDescription('');
    setTargetDate('');
    setEditId(null);
    fetchGoals();
    setLoading(false);
  };

  const handleDelete = async (id) => {
    await supabase.from('goals').delete().eq('id', id);
    fetchGoals();
  };

  const handleEdit = (goal) => {
    setEditId(goal.id);
    setTitle(goal.title);
    setDescription(goal.description);
    setTargetDate(goal.target_date);
  };

  const markAsDone = async (id) => {
    await supabase.from('goals').update({ status: 'Completed', completion_percentage: 100 }).eq('id', id);
    fetchGoals();
  };

  return (
    <div className="p-4 max-w-3xl mx-auto">
       <Button
            variant="outline"
            onClick={() => navigate("/IndividualDashboard")}
            className="fixed top-4 left-4 z-50 bg-white shadow-md hover:bg-muted text-sm"
           >
            ← Back to Dashboard
          </Button><br></br><br></br>
      
      <Card className="mb-6">
        <CardHeader>
          <CardTitle>{editId ? 'Edit Goal' : 'Add a New Goal'}</CardTitle>
        </CardHeader>
        <CardContent>
          <form onSubmit={handleSubmit} className="space-y-4">
            <Input
              type="text"
              placeholder="Goal title"
              value={title}
              onChange={(e) => setTitle(e.target.value)}
              required
            />
            <Textarea
              placeholder="Description (optional)"
              value={description}
              onChange={(e) => setDescription(e.target.value)}
            />
            <div>
              <label className="block mb-1 font-medium">Target Date</label>
              <Input
                type="date"
                value={targetDate}
                onChange={(e) => setTargetDate(e.target.value)}
              />
            </div>
            <Button type="submit" className="w-full" disabled={loading}>
              {loading ? 'Saving...' : editId ? 'Update Goal' : 'Add Goal'}
            </Button>
          </form>
        </CardContent>
      </Card>

      <div className="space-y-4">
        {goals.map((goal) => (
          <Card key={goal.id}>
            <CardHeader>
              <CardTitle className="flex justify-between">
                {goal.title}
                <span
                  className={`text-sm font-semibold ${
                    goal.status === 'Completed' ? 'text-green-600' : 'text-yellow-600'
                  }`}
                >
                  {goal.status}
                </span>
              </CardTitle>
            </CardHeader>
            <CardContent>
              <p className="text-muted-foreground mb-2">{goal.description}</p>
              <p className="text-sm mb-4">Target: {goal.target_date}</p>
              <div className="flex gap-2 flex-wrap">
                <Button size="sm" onClick={() => handleEdit(goal)}>Edit</Button>
                <Button size="sm" variant="destructive" onClick={() => handleDelete(goal.id)}>Delete</Button>
                {goal.status !== 'Completed' && (
                  <Button size="sm" onClick={() => markAsDone(goal.id)}>Mark as Done</Button>
                )}
              </div>
            </CardContent>
          </Card>
        ))}
      </div>
    </div>
  );
}
