import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Progress } from '@/components/ui/progress';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { ArrowLeft, Target, TrendingUp, Calendar, Plus, CheckCircle, Clock, AlertCircle, Trophy, Lightbulb, Users, BookOpen, Edit, Trash2, MoreVertical } from 'lucide-react';
import { useNavigate } from "react-router-dom";
import { supabase } from '@/components/supabaseClient';
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";

interface CareerGoalsProps {
  onBack: () => void;
}

interface Milestone {
  name: string;
  completed: boolean;
  date: string;
}

interface CareerGoal {
  id: string;
  title: string;
  description: string;
  category: string;
  priority: string;
  status: string;
  progress: number;
  deadline: string;
  milestones?: string[];
  skills: string[];
  estimatedCompletion: string;
  planning?: string;
  created_at?: string;
}

const goalTemplates = [
  {
    category: 'Career Advancement',
    title: 'Get Promoted to [Position]',
    description: 'Advance to the next level in your career with increased responsibilities',
    suggestedMilestones: ['Identify required skills', 'Complete relevant training', 'Take on leadership projects', 'Build relationships with stakeholders']
  },
  {
    category: 'Skill Development',
    title: 'Master [Technology/Skill]',
    description: 'Develop deep expertise in a specific technology or skill area',
    suggestedMilestones: ['Complete online courses', 'Build practice projects', 'Get certified', 'Apply skills in real projects']
  },
  {
    category: 'Entrepreneurship',
    title: 'Launch Side Business',
    description: 'Start and grow a profitable side business or consulting practice',
    suggestedMilestones: ['Validate business idea', 'Build MVP', 'Acquire first customers', 'Scale operations']
  },
  {
    category: 'Education',
    title: 'Complete Advanced Degree',
    description: 'Earn an advanced degree to enhance career prospects',
    suggestedMilestones: ['Research programs', 'Complete applications', 'Secure funding', 'Graduate successfully']
  }
];

export function CareerGoals({ onBack }: CareerGoalsProps) {
  const navigate = useNavigate();
  const [selectedGoal, setSelectedGoal] = useState<string | null>(null);
  const [showCreateGoal, setShowCreateGoal] = useState(false);
  const [showEditGoal, setShowEditGoal] = useState(false);
  const [careerGoals, setCareerGoals] = useState<CareerGoal[]>([]);
  const [loading, setLoading] = useState(true);
  const [editingGoal, setEditingGoal] = useState<CareerGoal | null>(null);
  const [formData, setFormData] = useState({
    title: '',
    description: '',
    category: 'Career Advancement',
    priority: 'High',
    status: 'Planning',
    deadline: '',
    planning: '',
    milestones: '',
    skills: '',
    progress: '',
    estimatedCompletion: ''
  });

  // Fetch career goals from Supabase
  useEffect(() => {
    fetchCareerGoals();
  }, []);

  const fetchCareerGoals = async () => {
    try {
      setLoading(true);
      const { data, error } = await supabase
        .from('career_goals')
        .select('*')
        .order('created_at', { ascending: false });

      if (error) {
        throw error;
      }

      if (data) {
        // Parse milestones and skills from JSON strings
        const goals = data.map(goal => ({
          ...goal,
          milestones: goal.milestones ? JSON.parse(goal.milestones) : [],
          skills: goal.skills ? JSON.parse(goal.skills) : []
        }));
        setCareerGoals(goals);
      }
    } catch (error) {
      console.error('Error fetching career goals:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleCreateGoal = async () => {
    try {
      const newGoal = {
        title: formData.title,
        description: formData.description,
        category: formData.category,
        priority: formData.priority,
        status: formData.status,
        progress: formData.progress,
        deadline: formData.deadline,
        planning: formData.planning,
        milestones: JSON.stringify(formData.milestones.split(',').map(milestone => milestone.trim()).filter(milestone => milestone)), 
        skills: JSON.stringify(formData.skills.split(',').map(skill => skill.trim()).filter(skill => skill)),
        estimatedCompletion: formData.estimatedCompletion
      };

      const { data, error } = await supabase
        .from('career_goals')
        .insert([newGoal])
        .select();

      if (error) {
        throw error;
      }

      if (data) {
        await fetchCareerGoals();
        setShowCreateGoal(false);
        setFormData({
          title: '',
          description: '',
          category: 'Career Advancement',
          priority: 'High',
          status: 'Planning',
          deadline: '',
          planning: '',
          milestones: '',
          progress: '',
          skills: '',
          estimatedCompletion: ''
        });
      }
    } catch (error) {
      console.error('Error creating goal:', error);
    }
  };

  const handleEditGoal = async () => {
    if (!editingGoal) return;

    try {
      const updatedGoal = {
        title: formData.title,
        description: formData.description,
        category: formData.category,
        priority: formData.priority,
        status: formData.status,
        deadline: formData.deadline,
        progress: formData.progress,
        planning: formData.planning,
        milestones: JSON.stringify(formData.milestones.split(',').map(milestone => milestone.trim()).filter(milestone => milestone)),
        skills: JSON.stringify(formData.skills.split(',').map(skill => skill.trim()).filter(skill => skill)),
        estimatedCompletion: formData.estimatedCompletion
      };

      const { error } = await supabase
        .from('career_goals')
        .update(updatedGoal)
        .eq('id', editingGoal.id);

      if (error) {
        throw error;
      }

      await fetchCareerGoals();
      setShowEditGoal(false);
      setEditingGoal(null);
    } catch (error) {
      console.error('Error updating goal:', error);
    }
  };

  const handleDeleteGoal = async (id: string) => {
    if (!confirm('Are you sure you want to delete this career goal?')) {
      return;
    }

    try {
      const { error } = await supabase
        .from('career_goals')
        .delete()
        .eq('id', id);

      if (error) {
        throw error;
      }

      await fetchCareerGoals();
    } catch (error) {
      console.error('Error deleting goal:', error);
    }
  };

  const openEditDialog = (goal: CareerGoal) => {
    setEditingGoal(goal);
    setFormData({
      title: goal.title,
      description: goal.description,
      category: goal.category,
      priority: goal.priority,
      status: goal.status,
      deadline: goal.deadline,
      progress: goal.progress,
      planning: goal.planning || '',
      milestones: goal.milestones.join(', '),
      skills: goal.skills.join(', '),
      estimatedCompletion: goal.estimatedCompletion
    });
    setShowEditGoal(true);
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'Completed': return 'bg-green-100 text-green-800';
      case 'In Progress': return 'bg-blue-100 text-blue-800';
      case 'Active': return 'bg-green-100 text-green-800';
      case 'Planning': return 'bg-yellow-100 text-yellow-800';
      case 'On Hold': return 'bg-gray-100 text-gray-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case 'High': return 'bg-red-100 text-red-800';
      case 'Medium': return 'bg-yellow-100 text-yellow-800';
      case 'Low': return 'bg-green-100 text-green-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getCategoryIcon = (category: string) => {
    switch (category) {
      case 'Career Advancement': return Trophy;
      case 'Skill Development': return BookOpen;
      case 'Education': return BookOpen;
      case 'Personal Branding': return Users;
      case 'Entrepreneurship': return Lightbulb;
      default: return Target;
    }
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric'
    });
  };

  const calculateDaysRemaining = (deadline: string) => {
    const today = new Date();
    const deadlineDate = new Date(deadline);
    const diffTime = deadlineDate.getTime() - today.getTime();
    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
    return diffDays;
  };

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="border-b bg-card/50 backdrop-blur-sm sticky top-0 z-50">
        <div className="container mx-auto px-6 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-4">
              <Button variant="ghost" size="sm"  onClick={() => navigate("/CareerDevelopment")}
                className="w-4 h-4 mr-2">
                <ArrowLeft className="w-4 h-4" />
             </Button>
              <div className="w-10 h-10 rounded-lg bg-gradient-primary flex items-center justify-center">
                <Target className="w-6 h-6 text-white" />
              </div>
              <div>
                <h1 className="text-2xl font-bold bg-gradient-hero bg-clip-text text-transparent">
                  Career Goals
                </h1>
                <p className="text-sm text-muted-foreground">Set, track, and achieve your professional milestones</p>
              </div>
            </div>
            <div className="flex items-center space-x-2">
              <Badge variant="secondary" className="bg-violet-800 text-white">
                {careerGoals.length} Active Goals
              </Badge>
              <Dialog open={showCreateGoal} onOpenChange={setShowCreateGoal}>
                <DialogTrigger asChild>
                  <Button size="sm">
                    <Plus className="w-4 h-4 mr-2" />
                    New Goal
                  </Button>
                </DialogTrigger>
                <DialogContent className="max-w-2xl">
                  <DialogHeader>
                    <DialogTitle>Create New Career Goal</DialogTitle>
                    <DialogDescription>
                      Set a new professional goal with milestones and tracking
                    </DialogDescription>
                  </DialogHeader>
                  <div className="space-y-4">
                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <Label htmlFor="title">Goal Title</Label>
                        <Input 
                          id="title" 
                          placeholder="Enter goal title" 
                          value={formData.title}
                          onChange={(e) => setFormData({...formData, title: e.target.value})}
                        />
                      </div>
                      <div>
                        <Label htmlFor="category">Category</Label>
                        <Select 
                          value={formData.category}
                          onValueChange={(value) => setFormData({...formData, category: value})}
                        >
                          <SelectTrigger>
                            <SelectValue placeholder="Select category" />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="Career Advancement">Career Advancement</SelectItem>
                            <SelectItem value="Skill Development">Skill Development</SelectItem>
                            <SelectItem value="Education">Education</SelectItem>
                            <SelectItem value="Entrepreneurship">Entrepreneurship</SelectItem>
                            <SelectItem value="Personal Branding">Personal Branding</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>
                    </div>
                    <div>
                      <Label htmlFor="description">Description</Label>
                      <Textarea 
                        id="description" 
                        placeholder="Describe your goal in detail" 
                        value={formData.description}
                        onChange={(e) => setFormData({...formData, description: e.target.value})}
                      />
                    </div>
                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <Label htmlFor="priority">Priority</Label>
                        <Select 
                          value={formData.priority}
                          onValueChange={(value) => setFormData({...formData, priority: value})}
                        >
                          <SelectTrigger>
                            <SelectValue placeholder="Select priority" />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="High">High</SelectItem>
                            <SelectItem value="Medium">Medium</SelectItem>
                            <SelectItem value="Low">Low</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>
                      <div>
                        <Label htmlFor="deadline">Target Deadline</Label>
                        <Input 
                          id="deadline" 
                          type="date" 
                          value={formData.deadline}
                          onChange={(e) => setFormData({...formData, deadline: e.target.value})}
                        />
                      </div>
                    </div>
                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <Label htmlFor="status">Status</Label>
                        <Select 
                          value={formData.status}
                          onValueChange={(value) => setFormData({...formData, status: value})}
                        >
                          <SelectTrigger>
                            <SelectValue placeholder="Select status" />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="Planning">Planning</SelectItem>
                            <SelectItem value="In Progress">In Progress</SelectItem>
                            <SelectItem value="Active">Active</SelectItem>
                            <SelectItem value="On Hold">On Hold</SelectItem>
                            <SelectItem value="Completed">Completed</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>
                      <div>
                        <Label htmlFor="estimatedCompletion">Estimated Completion</Label>
                        <Input 
                          id="estimatedCompletion" 
                          placeholder="e.g., 6 months" 
                          value={formData.estimatedCompletion}
                          onChange={(e) => setFormData({...formData, estimatedCompletion: e.target.value})}
                        />
                      </div>
                        <div className="space-y-2">
                        <label className="text-sm font-medium">Progress (0-100)</label>
                        <Input
                        name="progress"
                        type="number"
                        min="0"
                        max="100"
                        value={FormData.progress}
                        onChange={(e) => setFormData({...formData, progress: e.target.value})}/>
                        </div>
                    <div>
                      <Label htmlFor="milestones">Milestones (comma-separated)</Label>
                      <Input 
                        id="milestones" 
                        placeholder="e.g., Job, Promotion, Certification" 
                        value={formData.milestones}
                        onChange={(e) => setFormData({...formData, milestones: e.target.value})}
                      />
                    </div>
                    </div>

                    <div>
                      <Label htmlFor="planning">Planning Details</Label>
                      <Textarea 
                        id="planning" 
                        placeholder="Describe your plan to achieve this goal..." 
                        value={formData.planning}
                        onChange={(e) => setFormData({...formData, planning: e.target.value})}
                      />
                    </div>

                    <div>
                      <Label htmlFor="skills">Required Skills (comma-separated)</Label>
                      <Input 
                        id="skills" 
                        placeholder="e.g., React, Node.js, Leadership" 
                        value={formData.skills}
                        onChange={(e) => setFormData({...formData, skills: e.target.value})}
                      />
                    </div>
                    <div className="flex justify-end space-x-2">
                      <Button variant="outline" onClick={() => setShowCreateGoal(false)}>
                        Cancel
                      </Button>
                      <Button onClick={handleCreateGoal} disabled={!formData.title || !formData.description}>
                        Create Goal
                      </Button>
                    </div>
                  </div>
                </DialogContent>
              </Dialog>

              {/* Edit Dialog */}
              <Dialog open={showEditGoal} onOpenChange={setShowEditGoal}>
                <DialogContent className="max-w-2xl">
                  <DialogHeader>
                    <DialogTitle>Edit Career Goal</DialogTitle>
                    <DialogDescription>
                      Update your career goal details
                    </DialogDescription>
                  </DialogHeader>
                  <div className="space-y-4">
                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <Label htmlFor="edit-title">Goal Title</Label>
                        <Input 
                          id="edit-title" 
                          value={formData.title}
                          onChange={(e) => setFormData({...formData, title: e.target.value})}
                        />
                      </div>
                      <div>
                        <Label htmlFor="edit-category">Category</Label>
                        <Select 
                          value={formData.category}
                          onValueChange={(value) => setFormData({...formData, category: value})}
                        >
                          <SelectTrigger>
                            <SelectValue placeholder="Select category" />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="Career Advancement">Career Advancement</SelectItem>
                            <SelectItem value="Skill Development">Skill Development</SelectItem>
                            <SelectItem value="Education">Education</SelectItem>
                            <SelectItem value="Entrepreneurship">Entrepreneurship</SelectItem>
                            <SelectItem value="Personal Branding">Personal Branding</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>
                    </div>
                    <div>
                      <Label htmlFor="edit-description">Description</Label>
                      <Textarea 
                        id="edit-description" 
                        value={formData.description}
                        onChange={(e) => setFormData({...formData, description: e.target.value})}
                      />
                    </div>
                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <Label htmlFor="edit-priority">Priority</Label>
                        <Select 
                          value={formData.priority}
                          onValueChange={(value) => setFormData({...formData, priority: value})}
                        >
                          <SelectTrigger>
                            <SelectValue placeholder="Select priority" />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="High">High</SelectItem>
                            <SelectItem value="Medium">Medium</SelectItem>
                            <SelectItem value="Low">Low</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>
                      <div>
                        <Label htmlFor="edit-deadline">Target Deadline</Label>
                        <Input 
                          id="edit-deadline" 
                          type="date" 
                          value={formData.deadline}
                          onChange={(e) => setFormData({...formData, deadline: e.target.value})}
                        />
                      </div>
                    </div>
                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <Label htmlFor="edit-status">Status</Label>
                        <Select 
                          value={formData.status}
                          onValueChange={(value) => setFormData({...formData, status: value})}
                        >
                          <SelectTrigger>
                            <SelectValue placeholder="Select status" />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="Planning">Planning</SelectItem>
                            <SelectItem value="In Progress">In Progress</SelectItem>
                            <SelectItem value="Active">Active</SelectItem>
                            <SelectItem value="On Hold">On Hold</SelectItem>
                            <SelectItem value="Completed">Completed</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>


                      <div className="space-y-2">
                        <label className="text-sm font-medium">Progress (0-100)</label>
                        <Input
                        name="progress"
                        type="number"
                        min="0"
                        max="100"
                        value={FormData.progress}
                        onChange={(e) => setFormData({...formData, progress: e.target.value})}/>
                        </div>
                      <div>
                        <Label htmlFor="edit-estimatedCompletion">Estimated Completion</Label>
                        <Input 
                          id="edit-estimatedCompletion" 
                          value={formData.estimatedCompletion}
                          onChange={(e) => setFormData({...formData, estimatedCompletion: e.target.value})}
                        />
                      </div>

                      <div>
                      <Label htmlFor="milestones">Milestones</Label>
                      <Textarea 
                        id="milestones"
                        value={formData.milestones}
                        onChange={(e) => setFormData({...formData, milestones: e.target.value})}
                      />
                    </div>
                    </div>

                    
                    <div>
                      <Label htmlFor="edit-planning">Planning Details</Label>
                      <Textarea 
                        id="edit-planning" 
                        value={formData.planning}
                        onChange={(e) => setFormData({...formData, planning: e.target.value})}
                      />
                    </div>
                    <div>
                      <Label htmlFor="edit-skills">Required Skills (comma-separated)</Label>
                      <Input 
                        id="edit-skills" 
                        value={formData.skills}
                        onChange={(e) => setFormData({...formData, skills: e.target.value})}
                      />
                    </div>
                    <div className="flex justify-end space-x-2">
                      <Button variant="outline" onClick={() => setShowEditGoal(false)}>
                        Cancel
                      </Button>
                      <Button onClick={handleEditGoal} disabled={!formData.title || !formData.description}>
                        Update Goal
                      </Button>
                    </div>
                  </div>
                </DialogContent>
              </Dialog>
            </div>
          </div>
        </div>
      </div>

      {/* Main Content */}
      <div className="container mx-auto px-6 py-8">
        <Tabs defaultValue="active" className="w-full">
          <TabsList className="grid w-full grid-cols-2">
            <TabsTrigger value="active">Active Goals</TabsTrigger>
            <TabsTrigger value="templates">Templates</TabsTrigger>
          </TabsList>

          {/* Active Goals Tab */}
          <TabsContent value="active" className="space-y-6">
            {loading ? (
              <div className="text-center py-8">
                <p>Loading career goals...</p>
              </div>
            ) : (
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                {careerGoals
                  .filter(goal => goal.status === 'In Progress' || goal.status === 'Active')
                  .map((goal) => {
                  const CategoryIcon = getCategoryIcon(goal.category);
                  const daysRemaining = calculateDaysRemaining(goal.deadline);
                  
                  return (
                    <Card 
                      key={goal.id} 
                      className="cursor-pointer hover:shadow-strong transition-all duration-300"
                      onClick={() => setSelectedGoal(selectedGoal === goal.id ? null : goal.id)}
                    >
                      <CardHeader>
                        <div className="flex items-start justify-between">
                          <div className="flex items-center space-x-3">
                            <div className="w-10 h-10 bg-gradient-primary rounded-lg flex items-center justify-center">
                              <CategoryIcon className="w-5 h-5 text-white" />
                            </div>
                            <div>
                              <CardTitle className="text-lg">{goal.title}</CardTitle>
                              <div className="flex items-center space-x-2 mt-1">
                                <Badge variant="outline" className={getStatusColor(goal.status)}>
                                  {goal.status}
                                </Badge>
                                <Badge variant="outline" className={getPriorityColor(goal.priority)}>
                                  {goal.priority}
                                </Badge>
                              </div>
                            </div>
                          </div>
                          <div className="flex items-center space-x-2">
                            <div className="text-right">
                              <p className="text-sm font-medium">{goal.progress}%</p>
                              <p className="text-xs text-muted-foreground">Complete</p>
                            </div>
                            <DropdownMenu>
                              <DropdownMenuTrigger asChild>
                                <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                                  <MoreVertical className="h-4 w-4" />
                                </Button>
                              </DropdownMenuTrigger>
                              <DropdownMenuContent align="end">
                                <DropdownMenuItem onClick={() => openEditDialog(goal)}>
                                  <Edit className="w-4 h-4 mr-2" />
                                  Edit
                                </DropdownMenuItem>
                                <DropdownMenuItem 
                                  onClick={() => handleDeleteGoal(goal.id)}
                                  className="text-red-600"
                                >
                                  <Trash2 className="w-4 h-4 mr-2" />
                                  Delete
                                </DropdownMenuItem>
                              </DropdownMenuContent>
                            </DropdownMenu>
                          </div>
                        </div>
                      </CardHeader>
                      <CardContent className="space-y-4">
                        <CardDescription>{goal.description}</CardDescription>
                        
                        <div>
                          <div className="flex items-center justify-between mb-1">
                            <span className="text-sm font-medium">Progress</span>
                            <span className="text-sm">{goal.progress}%</span>
                          </div>
                          <Progress value={goal.progress} className="h-2" />
                        </div>

                        <div className="flex items-center justify-between text-sm">
                          <div className="flex items-center text-muted-foreground">
                            <Calendar className="w-4 h-4 mr-1" />
                            {formatDate(goal.deadline)}
                          </div>
                          <div className={`flex items-center ${daysRemaining < 30 ? 'text-red-500' : 'text-muted-foreground'}`}>
                            <Clock className="w-4 h-4 mr-1" />
                            {daysRemaining > 0 ? `${daysRemaining} days left` : 'Overdue'}
                          </div>
                        </div>

                        {selectedGoal === goal.id && (
                          <div className="mt-4 pt-4 border-t space-y-3">
                            <h4 className="font-medium">Milestones</h4>
                            <div className="space-y-2">
                              <div className="flex flex-wrap gap-1">
                                <p className="text-sm text-muted-foreground">{goal.milestones}</p>
                                
                              </div>
                              
                            </div>
                            
                            {goal.planning && (
                              <div className="pt-2">
                                <p className="text-sm font-medium mb-2">Planning</p>
                                <p className="text-sm text-muted-foreground">{goal.planning}</p>
                              </div>
                            )}
                            
                            <div className="pt-2">
                              <p className="text-sm font-medium mb-2">Required Skills</p>
                              <div className="flex flex-wrap gap-1">
                                {goal.skills.map((skill) => (
                                  <Badge key={skill} variant="secondary" className="text-xs">
                                    {skill}
                                  </Badge>
                                ))}
                              </div>
                            </div>
                          </div>
                        )}
                      </CardContent>
                    </Card>
                  );
                })}
              </div>
            )}
          </TabsContent>

          {/* Planning Tab */}
          <TabsContent value="planning" className="space-y-6">
            {loading ? (
              <div className="text-center py-8">
                <p>Loading career goals...</p>
              </div>
            ) : (
              <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                {careerGoals
                  .filter(goal => goal.status === 'Planning')
                  .map((goal) => {
                  const CategoryIcon = getCategoryIcon(goal.category);
                  
                  return (
                    <Card key={goal.id} className="hover:shadow-strong transition-all duration-300">
                      <CardHeader>
                        <div className="flex items-start justify-between">
                          <div className="flex items-center space-x-3">
                            <div className="w-10 h-10 bg-gradient-primary rounded-lg flex items-center justify-center">
                              <CategoryIcon className="w-5 h-5 text-white" />
                            </div>
                            <div>
                              <CardTitle className="text-lg">{goal.title}</CardTitle>
                              <Badge variant="outline" className={getStatusColor(goal.status)}>
                                {goal.status}
                              </Badge>
                            </div>
                          </div>
                          <div className="flex items-center space-x-2">
                            <div className="text-right">
                              <p className="text-sm text-muted-foreground">Est. {goal.estimatedCompletion}</p>
                            </div>
                            <DropdownMenu>
                              <DropdownMenuTrigger asChild>
                                <Button variant="ghost" size="sm" className="h-8 w-8 p-0">
                                  <MoreVertical className="h-4 w-4" />
                                </Button>
                              </DropdownMenuTrigger>
                              <DropdownMenuContent align="end">
                                <DropdownMenuItem onClick={() => openEditDialog(goal)}>
                                  <Edit className="w-4 h-4 mr-2" />
                                  Edit
                                </DropdownMenuItem>
                                <DropdownMenuItem 
                                  onClick={() => handleDeleteGoal(goal.id)}
                                  className="text-red-600"
                                >
                                  <Trash2 className="w-4 h-4 mr-2" />
                                  Delete
                                </DropdownMenuItem>
                              </DropdownMenuContent>
                            </DropdownMenu>
                          </div>
                        </div>
                      </CardHeader>
                      <CardContent className="space-y-4">
                        <CardDescription>{goal.description}</CardDescription>
                        
                        {goal.planning && (
                          <div>
                            <p className="text-sm font-medium mb-2">Planning</p>
                            <p className="text-sm text-muted-foreground">{goal.planning}</p>
                          </div>
                        )}
                        
                        <div className="flex items-center justify-between text-sm">
                          <div className="flex items-center text-muted-foreground">
                            <Calendar className="w-4 h-4 mr-1" />
                            Target: {formatDate(goal.deadline)}
                          </div>
                          <Button variant="outline" size="sm" onClick={() => openEditDialog(goal)}>
                            Update Plan
                          </Button>
                        </div>
                      </CardContent>
                    </Card>
                  );
                })}
              </div>
            )}
          </TabsContent>

          {/* Completed Tab */}
          <TabsContent value="completed" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Completed Goals</CardTitle>
                <CardDescription>
                  Celebrate your achievements and track your professional growth
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="text-center py-8">
                  <Trophy className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
                  <p className="text-muted-foreground">No completed goals yet.</p>
                  <p className="text-sm text-muted-foreground">Your achievements will appear here once you complete your goals.</p>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Templates Tab */}
          <TabsContent value="templates" className="space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              {goalTemplates.map((template, index) => {
                const CategoryIcon = getCategoryIcon(template.category);
                
                return (
                  <Card key={index} className="hover:shadow-strong transition-all duration-300">
                    <CardHeader>
                      <div className="flex items-center space-x-3">
                        <div className="w-10 h-10 bg-gradient-primary rounded-lg flex items-center justify-center">
                          <CategoryIcon className="w-5 h-5 text-white" />
                        </div>
                        <div>
                          <CardTitle className="text-lg">{template.title}</CardTitle>
                          <Badge variant="secondary">{template.category}</Badge>
                        </div>
                      </div>
                    </CardHeader>
                    <CardContent className="space-y-4">
                      <CardDescription>{template.description}</CardDescription>
                      
                      <div>
                        <p className="text-sm font-medium mb-2">Suggested Milestones:</p>
                        <ul className="text-sm text-muted-foreground space-y-1">
                          {template.suggestedMilestones.map((milestone, idx) => (
                            <li key={idx} className="flex items-start">
                              <span className="w-1.5 h-1.5 bg-primary rounded-full mt-2 mr-2 flex-shrink-0" />
                              {milestone}
                            </li>
                          ))}
                        </ul>
                      </div>
                      
                      <Button className="w-full" onClick={() => setShowCreateGoal(true)}>
                        Use This Template
                      </Button>
                    </CardContent>
                  </Card>
                );
              })}
            </div>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
}