import React, { useEffect, useState } from "react";
import { supabase } from "@/components/supabaseClient";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { useToast } from "@/components/ui/use-toast";
import { useUser } from "@supabase/auth-helpers-react";
import { useNavigate } from 'react-router-dom';
import { utils, writeFile } from "xlsx";


import {
  BarChart, Bar, XAxis, YAxis, Tooltip, PieChart, Pie, Cell, ResponsiveContainer
} from 'recharts';

const calculateDifference = (projected: number, actual: number): number => {
  return (projected || 0) - (actual || 0);
};


export default function BudgetTracker() {
  const navigate = useNavigate();
  
  const [startingBudget, setStartingBudget] = useState(0);
  const [endingBudget, setEndingBudget] = useState(0);
  const [startingSaving, setStartingSaving] = useState(0);
  const [endingSaving, setEndingSaving] = useState(0);
  const [isEditingBudget, setIsEditingBudget] = useState(false);
  const [isEditingSaving, setIsEditingSaving] = useState(false);
  const [isEditingGoal, setIsEditingGoal] = useState(false);
  const [editingGoalId, setEditingGoalId] = useState<string | number | null>(null);


  const [savings_goals, setGoals] = useState([]);
  const [newGoal, setNewGoal] = useState({
    id: "",
    name: "",
    target_date: "",
    description: "",
    monthly_goal: 0,
    current_amount: 0,
    goal_balance: "",
    target_amount: 0,
    created_at: "",
    user_id: "",
  });

  const [expenses, setExpenses] = useState([
    { dueDate: "", paid: false, description: "", projected: 0, actual: 0 },
   
    
  ]);

  const fetchGoals = async () => {
    const { data, error } = await supabase.from("savings_goals").select("*");
    if (!error && data) {
      setGoals(data);
    }
  };

  const fetchBudgetBalance = async () => {
    const { data, error } = await supabase
      .from("budget_balance")
      .select("*")
      .eq("id", 1)
      .single();

    if (!error && data) {
      setStartingBudget(data.starting_budget || 0);
      setEndingBudget(data.ending_budget || 0);
    }
  };

  const fetchSavingBalance = async () => {
    const { data, error } = await supabase
      .from("saving_balance")
      .select("*")
      .eq("id", 1)
      .single();

    if (!error && data) {
      setStartingSaving(data.starting_saving || 0);
      setEndingSaving(data.ending_saving || 0);
    }
  };
  

  const handleAddGoal = async () => {
    try {
      const targetAmount = Number(newGoal.target_amount);
      const currentAmount = Number(newGoal.current_amount);
      const monthlyGoal = Number(newGoal.monthly_goal);
  
      if (isNaN(targetAmount) || isNaN(currentAmount)) {
        toast({
          title: "Error",
          description: "Please enter valid numeric amounts",
          variant: "destructive",
        });
        return;
      }

      if (!newGoal.name || !newGoal.target_date || targetAmount <= 0) {
      toast({
      title: "Validation Error",
      description: "Please fill all required fields and ensure amounts are valid.",
       variant: "destructive",
      });
      return;
      }
  
      const goal_balance = targetAmount - currentAmount;
  
      const { error, data } = await supabase.from("savings_goals").insert([
        {
          name: newGoal.name,
          target_date: newGoal.target_date,
          description: newGoal.description,
          monthly_goal: monthlyGoal,
          current_amount: currentAmount,
          target_amount: targetAmount,
          goal_balance,
          created_at: new Date().toISOString(),
          user_id: user?.id,
        },
      ]);
  
      if (error) throw error;
  
      toast({ title: "Success", description: "Goal saved successfully" });
  
      // Reset form
      setNewGoal({
        id: "",
        name: "",
        target_date: "",
        description: "",
        monthly_goal: 0,
        current_amount: 0,
        goal_balance: "",
        target_amount: 0,
        created_at: "",
        user_id: "",
      });
  
      fetchGoals();
    } catch (error) {
      console.error("Failed to save goal:", error);
      toast({
        title: "Error",
        description: "Failed to save Goal",
        variant: "destructive",
      });
    }
  };
  
  const handleEditGoal = (goal: any) => {
  setNewGoal(goal); // load goal into form
  setIsEditingGoal(true);
  setEditingGoalId(goal.id);
};

const handleUpdateGoal = async () => {
  try {
    const targetAmount = Number(newGoal.target_amount);
    const currentAmount = Number(newGoal.current_amount);
    const monthlyGoal = Number(newGoal.monthly_goal);

    const goal_balance = targetAmount - currentAmount;

    const { error } = await supabase
      .from("savings_goals")
      .update({
        name: newGoal.name,
        target_date: newGoal.target_date,
        description: newGoal.description,
        monthly_goal: monthlyGoal,
        current_amount: currentAmount,
        target_amount: targetAmount,
        goal_balance,
      })
      .eq("id", editingGoalId);

    if (error) throw error;

    toast({ title: "Success", description: "Goal updated successfully" });

    // Reset
    setNewGoal({
      id: "",
      name: "",
      target_date: "",
      description: "",
      monthly_goal: 0,
      current_amount: 0,
      goal_balance: "",
      target_amount: 0,
      created_at: "",
      user_id: "",
    });
    setIsEditingGoal(false);
    setEditingGoalId(null);

    fetchGoals();
  } catch (error: any) {
    console.error("Failed to update goal:", error);
    toast({
      title: "Error",
      description: error?.message || "Failed to update goal",
      variant: "destructive",
    });
  }
};


  const handleDeleteGoal = async (id: string | number) => {
  const confirmDelete = window.confirm("Are you sure you want to delete this goal?");
  if (!confirmDelete) return;

  try {
    const { error } = await supabase
      .from("savings_goals")
      .delete()
      .eq("id", id);

    if (error) throw error;

    toast({ title: "Success", description: "Goal deleted successfully" });
    fetchGoals(); // Refresh goal list
  } catch (error: any) {
    console.error("Failed to delete goal:", error);
    toast({
      title: "Error",
      description: error?.message || "Failed to delete goal",
      variant: "destructive",
    });
  }
};


  const handleSaveBudget = async () => {
    try {
      const { error } = await supabase.from("budget_balance").upsert({
        id: 1,
        starting_budget: startingBudget,
        ending_budget: endingBudget,
        updated_at: new Date().toISOString(),
      });

      if (error) throw error;

      toast({ title: "Success", description: "Budget balance saved successfully" });
      setIsEditingBudget(false);
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to save budget balance",
        variant: "destructive",
      });
    }
  };

  const handleSaveSaving = async () => {
    try {
      const { error } = await supabase.from("saving_balance").upsert({
        id: 1,
        starting_saving: startingSaving,
        ending_saving: endingSaving,
        updated_at: new Date().toISOString(),
      });

      if (error) throw error;

      toast({ title: "Success", description: "Saving balance saved successfully" });
      setIsEditingSaving(false);
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to save saving balance",
        variant: "destructive",
      });
    }
  };

  useEffect(() => {
    fetchBudgetBalance();
    fetchSavingBalance();
    fetchGoals();
  }, []);

  // State and user

const { toast } = useToast();


// Fetch on load
useEffect(() => {
  const fetchExpenses = async () => {
    const { data, error } = await supabase
      .from("monthly_expenses")
      .select("*")
      .eq("user_id", user?.id)
      .order("due_date", { ascending: true });

    if (error) console.error("Fetch error:", error);
    else setExpenses(data);
  };
});

// Handlers
const handleAddExpense = () => {
  setExpenses([
    ...expenses,
    {
      id: null,
      due_date: "",
      paid: false,
      description: "",
      projected: 0,
      actual: 0,
      isNew: true,
    },
  ]);
};

const handleSave = async (item, idx) => {
  const errors: Record<number, string> ={};
  const expenseData = {
    due_date: item.due_date,
    paid: item.paid,
    description: item.description,
    projected: item.projected,
    actual: item.actual,
    difference: item.difference,
  };

  if (item.id) {
    const { error } = await supabase
      .from("monthly_expenses")
      .update(expenseData)
      .eq("id", item.id);

    if (error) toast({ title: "Update error", description: error.message });
    else toast({ title: "Expense updated" });
  } else {
    const { data, error } = await supabase
      .from("monthly_expenses")
      .insert([expenseData])
      .select();

    if (error) toast({ title: "Insert error", description: error.message });
    else {
      const updated = [...expenses];
      updated[idx] = data[0];
      setExpenses(updated);
      toast({ title: "Expense added" });
    }
  }
};

const handleDelete = async (id, idx) => {
  if (id) {
    const { error } = await supabase
      .from("monthly_expenses")
      .delete()
      .eq("id", id);

    if (error) {
      toast({ title: "Delete error", description: error.message });
      return;
    }
  }
  const updated = expenses.filter((_, i) => i !== idx);
  setExpenses(updated);
  toast({ title: "Expense deleted" });
};

const exportToCSV = () => {
  const data = expenses.map((e) => ({
    "Due Date": e.due_date,
    Paid: e.paid ? "Yes" : "No",
    Description: e.description,
    Projected: e.projected,
    Actual: e.actual,
    Difference: e.projected - e.actual,
  }));
  const ws = utils.json_to_sheet(data);
  const wb = utils.book_new();
  utils.book_append_sheet(wb, ws, "MonthlyExpenses");
  writeFile(wb, "MonthlyExpenses.csv");
};

const formatCurrency = (num) =>
  num?.toLocaleString("en-US", { style: "currency", currency: "USD" });

const totalProjected = expenses.reduce((sum, e) => sum + (e.projected || 0), 0);
const totalActual = expenses.reduce((sum, e) => sum + (e.actual || 0), 0);
const totalDiff = totalProjected - totalActual;
const [expenseErrors, setExpenseErrors] = useState<Record<number, string>>({});


  return (
    
    <div className="flex min-h-screen">
      <Button
            variant="outline"
            onClick={() => navigate("/IndividualDashboard")}
            className="fixed top-4 left-4 z-50 bg-white shadow-md hover:bg-muted text-sm"
           >
            ← Back to Dashboard
          </Button><br></br><br></br>
    {/* Sidebar */}
    <aside className="w-full md:w-64 bg-gray-100 p-4 border-r">
      <h2 className="text-xl font-bold mb-4">Summary</h2>
      
      {/* Budget Balance */}
      <div className="mb-4">
        <h3 className="font-semibold">Budget Balance</h3>
        <p>Starting: {formatCurrency(startingBudget)}</p>
        <p>Ending: {formatCurrency(endingBudget)}</p>
      </div>

      {/* Saving Balance */}
      <div className="mb-4">
        <h3 className="font-semibold">Saving Balance</h3>
        <p>Starting: {formatCurrency(startingSaving)}</p>
        <p>Ending: {formatCurrency(endingSaving)}</p>
      </div>

      {/* Saving Goals */}
      <div className="mb-4">
        <h3 className="font-semibold">Saving Goals</h3>
        <p>Total Goals: {savings_goals.length}</p>
        <p>
          Total Target:{" "}
          {formatCurrency(
            savings_goals.reduce((acc, g) => acc + (g.target_amount || 0), 0)
          )}
        </p>
      </div>

      {/* Monthly Expenses */}
      <div className="mb-4">
        <h3 className="font-semibold">Monthly Expenses</h3>
        <p>Projected: {formatCurrency(totalProjected)}</p>
        <p>Actual: {formatCurrency(totalActual)}</p>
        <p>Difference: {formatCurrency(totalDiff)}</p>
      </div>
    </aside>

    {/* Main Content */}
    <main className="flex-1 p-4">
      <h1 className="text-2xl font-bold mb-4">Personal Budget Tracker</h1>
    <div className="p-4">
      <h1 className="text-2xl font-bold mb-4">Personal Budget Tracker</h1>

      {/* Budget Section */}
      <div className="mb-6">
        <div className="flex justify-between items-center mb-2">
          <h2 className="text-xl font-semibold">Budget Balance</h2>
          {isEditingBudget ? (
            <div className="space-x-2">
              <Button onClick={handleSaveBudget}>Save</Button>
              <Button variant="outline" onClick={() => setIsEditingBudget(false)}>Cancel</Button>
            </div>
          ) : (
            <Button onClick={() => setIsEditingBudget(true)}>Edit</Button>
          )}
        </div>
        <div className="grid grid-cols-2 gap-4">
          <div>
            <label>Starting Budget Balance</label>
            <Input
              type="number"
              value={startingBudget}
              onChange={e => setStartingBudget(Number(e.target.value))}
              disabled={!isEditingBudget}
            />
          </div>
          <div>
            <label>Ending Budget Balance</label>
            <Input
              type="number"
              value={endingBudget}
              onChange={e => setEndingBudget(Number(e.target.value))}
              disabled={!isEditingBudget}
            />
          </div>
        </div>
      </div>

      {/* Saving Section */}
      <div className="mb-6">
        <div className="flex justify-between items-center mb-2">
          <h2 className="text-xl font-semibold">Saving Balance</h2>
          {isEditingSaving ? (
            <div className="space-x-2">
              <Button onClick={handleSaveSaving}>Save</Button>
              <Button variant="outline" onClick={() => setIsEditingSaving(false)}>Cancel</Button>
            </div>
          ) : (
            <Button onClick={() => setIsEditingSaving(true)}>Edit</Button>
          )}
        </div>
        <div className="grid grid-cols-2 gap-4">
          <div>
            <label>Starting Saving Balance</label>
            <Input
              type="number"
              value={startingSaving}
              onChange={e => setStartingSaving(Number(e.target.value))}
              disabled={!isEditingSaving}
            />
          </div>
          <div>
            <label>Ending Saving Balance</label>
            <Input
              type="number"
              value={endingSaving}
              onChange={e => setEndingSaving(Number(e.target.value))}
              disabled={!isEditingSaving}
            />
          </div>
        </div>
      </div>

      {/* Saving Goals Section */}
      <h2 className="text-xl font-semibold mb-2">Saving Goals</h2>
      <table className="w-full border-collapse mb-4">
        <thead className="bg-blue-100">
          <tr>
            <th>Goal Name</th>
            <th>Goal Date</th>
            <th>Description</th>
            <th>Monthly Target</th>
            <th>Saved This Month</th>
            <th>Goal Balance</th>
            <th>Goal Target</th>
            <th>Actions</th>
          </tr>
        </thead>
        <tbody>
          {savings_goals.map((goal: any) => (
            <tr key={goal.id} className="border">
              <td>{goal.name}</td>
              <td>{goal.target_date}</td>
              <td>{goal.description}</td>
              <td>${goal.monthly_goal}</td>
              <td>${goal.current_amount}</td>
              <td>${goal.target_amount-goal.current_amount}</td>
              <td>${goal.target_amount}</td>
              <td>
                <Button variant="outline" className="mr-2" onClick={() => handleEditGoal(goal)}>Edit</Button>
                <Button variant="destructive" onClick={() => handleDeleteGoal(goal.id)}>Delete</Button>
              </td>
            </tr>
          ))}
        </tbody>
      </table>

      {/* Add New Goal */}
      <div className="mb-6">
        <h3 className="font-semibold mb-2">Add New Goal</h3>
        <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
          <Input placeholder="Goal Name" value={newGoal.name} onChange={e => setNewGoal({ ...newGoal, name: e.target.value })} />
          <Input type="date" value={newGoal.target_date} onChange={e => setNewGoal({ ...newGoal, target_date: e.target.value })} />
          <Textarea placeholder="Description" value={newGoal.description} onChange={e => setNewGoal({ ...newGoal, description: e.target.value })} />
          <Input placeholder="Monthly Goal" value={newGoal.monthly_goal} onChange={e => setNewGoal({ ...newGoal, monthly_goal: e.target.value })} />
          <Input placeholder="Saved this Month" value={newGoal.current_amount} onChange={e => setNewGoal({ ...newGoal, current_amount: e.target.value })} />
          <Input placeholder="Target Amount" value={newGoal.target_amount} onChange={e => setNewGoal({ ...newGoal, target_amount: e.target.value })} />
        </div>
          {isEditingGoal ? (
          <div className="space-x-2 mt-2">
          <Button onClick={handleUpdateGoal}>Update Goal</Button>
          <Button variant="outline" onClick={() => {
          setIsEditingGoal(false);
          setEditingGoalId(null);
          setNewGoal({
          id: "",
          name: "",
          target_date: "",
          description: "",
          monthly_goal: 0,
          current_amount: 0,
          goal_balance: "",
          target_amount: 0,
          created_at: "",
          user_id: "",
          });
          }}>
          Cancel
          </Button>
          </div>
         ) : (
          <Button className="mt-2" onClick={handleAddGoal}>Add Goal</Button>
         )}
      </div>

       <div className="mt-10">
       <h2 className="text-xl font-semibold mb-4">Monthly Expenses</h2>
       <div className="flex gap-2 mb-4">
       <Button onClick={handleAddExpense}>Add Expense</Button>
       <Button onClick={exportToCSV}>Export CSV</Button>
     </div>
     

      <table className="w-full border-collapse">
      <thead className="bg-blue-100 font-bold">
       <tr>
        <th>Due Date</th>
        <th>Paid</th>
        <th>Description</th>
        <th>Projected</th>
        <th>Actual</th>
        <th>Difference</th>
        <th>Action</th>
      </tr>
      </thead>
      <tbody>
      {expenses.map((item, idx) => (
        <tr key={idx} className="border">
          <td>
            <Input
              type="date"
              value={item.due_date || ""}
              onChange={(e) => {
                const updated = [...expenses];
                updated[idx].due_date = e.target.value;
                setExpenses(updated);
              }}
            />
          </td>
          <td>
            <input
              type="checkbox"
              checked={item.paid}
              onChange={(e) => {
                const updated = [...expenses];
                updated[idx].paid = e.target.checked;
                setExpenses(updated);
              }}
            />
          </td>
          <td>
            <Input
              type="text"
              value={item.description}
              onChange={(e) => {
                const updated = [...expenses];
                updated[idx].description = e.target.value;
                setExpenses(updated);
              }}
            />
          </td>
          <td>
            <Input
              type="number"
              value={item.projected}
              onChange={(e) => {
                const updated = [...expenses];
                updated[idx].projected = Number(e.target.value);
                setExpenses(updated);
              }}
            />
          </td>
          <td>
            <Input
              type="number"
              value={item.actual}
              onChange={(e) => {
                const updated = [...expenses];
                updated[idx].actual = Number(e.target.value);
                setExpenses(updated);
              }}
            />
          </td>
          <td>{formatCurrency(item.projected - item.actual)}</td>
          <td className="flex flex-col gap-1">
            <Button onClick={() => handleSave(item, idx)}>Save</Button>
            <Button
              variant="destructive"
              onClick={() => handleDelete(item.id, idx)}
            >
              Delete
            </Button>
          </td>
         </tr>
        ))}
       <tr className="font-bold bg-gray-100">
        <td colSpan={3}>TOTAL</td>
        <td>{formatCurrency(totalProjected)}</td>
        <td>{formatCurrency(totalActual)}</td>
        <td>{formatCurrency(totalDiff)}</td>
        <td></td>
       </tr>
       </tbody>
       </table>
      </div>
      </div>
      </main>
    </div>

)}