import { useState, useEffect } from "react";
import { createClient } from "@supabase/supabase-js";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Progress } from "@/components/ui/progress";
import { Badge } from "@/components/ui/badge";
import { DollarSign, TrendingUp, TrendingDown, Target, Plus, Trash2 } from "lucide-react";
import { format } from "date-fns";


interface Account {
  id: string;
  name: string;
  type: 'checking' | 'savings';
  starting_balance: number;
  current_balance: number;
}

interface Transaction {
  id: string;
  account_id: string;
  amount: number;
  description: string;
  category_id: string;
  date: string;
  type: 'income' | 'expense' | 'transfer';
}

interface BudgetCategory {
  id: string;
  name: string;
  is_fixed: boolean;
}

interface SavingsGoal {
  id: string;
  name: string;
  target_amount: number;
  current_amount: number;
  target_date?: string;
  monthly_goal?: number;
}

export const BudgetTracker = () => {
  const [accounts, setAccounts] = useState<Account[]>([]);
  const [transactions, setTransactions] = useState<Transaction[]>([]);
  const [categories, setCategories] = useState<BudgetCategory[]>([]);
  const [savingsGoals, setSavingsGoals] = useState<SavingsGoal[]>([]);
  const [loading, setLoading] = useState(true);
  
  // Form states
  const [newTransaction, setNewTransaction] = useState({
    account_id: '',
    amount: '',
    description: '',
    category_id: '',
    type: 'expense' as const,
    date: format(new Date(), 'yyyy-MM-dd')
  });

  const [newGoal, setNewGoal] = useState({
    name: "",
    target_amount: "",
    current_amount: "0",
    target_date: "",
    monthly_goal: ""
  });

  // Load initial data
  useEffect(() => {
    const fetchData = async () => {
      try {
        setLoading(true);
        
        // Fetch all data in parallel
        const [
          { data: accountsData },
          { data: transactionsData },
          { data: categoriesData },
          { data: goalsData }
        ] = await Promise.all([
          supabase.from('accounts').select('*'),
          supabase.from('transactions').select('*'),
          supabase.from('budget_categories').select('*'),
          supabase.from('savings_goals').select('*')
        ]);

        if (accountsData) setAccounts(accountsData);
        if (transactionsData) setTransactions(transactionsData);
        if (categoriesData) setCategories(categoriesData);
        if (goalsData) setSavingsGoals(goalsData);
        
      } catch (error) {
        console.error('Error fetching data:', error);
      } finally {
        setLoading(false);
      }
    };

    fetchData();

    // Set up real-time subscriptions
    const transactionsSubscription = supabase
      .channel('transactions_changes')
      .on('postgres_changes', {
        event: '*',
        schema: 'public',
        table: 'transactions'
      }, (payload) => {
        if (payload.eventType === 'INSERT') {
          setTransactions(prev => [...prev, payload.new as Transaction]);
        } else if (payload.eventType === 'UPDATE') {
          setTransactions(prev => prev.map(t => 
            t.id === payload.new.id ? payload.new as Transaction : t
          ));
        } else if (payload.eventType === 'DELETE') {
          setTransactions(prev => prev.filter(t => t.id !== payload.old.id));
        }
      })
      .subscribe();

    return () => {
      supabase.removeChannel(transactionsSubscription);
    };
  }, []);

  const addTransaction = async () => {
    if (!newTransaction.account_id || !newTransaction.amount || !newTransaction.description) {
      alert('Please fill all required fields');
      return;
    }

    try {
      const { data, error } = await supabase
        .from('transactions')
        .insert([{
          account_id: newTransaction.account_id,
          amount: parseFloat(newTransaction.amount) * (newTransaction.type === 'expense' ? -1 : 1),
          description: newTransaction.description,
          category_id: newTransaction.category_id,
          type: newTransaction.type,
          date: newTransaction.date
        }])
        .select();

      if (error) throw error;

      if (data) {
        // Update account balance
        const account = accounts.find(a => a.id === newTransaction.account_id);
        if (account) {
          const amount = parseFloat(newTransaction.amount) * (newTransaction.type === 'expense' ? -1 : 1);
          await supabase
            .from('accounts')
            .update({ current_balance: account.current_balance + amount })
            .eq('id', account.id);
        }

        // Reset form
        setNewTransaction({
          account_id: '',
          amount: '',
          description: '',
          category_id: '',
          type: 'expense',
          date: format(new Date(), 'yyyy-MM-dd')
        });
      }
    } catch (error) {
      console.error('Error adding transaction:', error);
    }
  };

  const removeTransaction = async (id: string) => {
    try {
      // First get the transaction to update the account balance
      const { data: transaction } = await supabase
        .from('transactions')
        .select('*')
        .eq('id', id)
        .single();

      if (!transaction) return;

      // Delete the transaction
      const { error } = await supabase
        .from('transactions')
        .delete()
        .eq('id', id);

      if (error) throw error;

      // Update account balance
      const account = accounts.find(a => a.id === transaction.account_id);
      if (account) {
        await supabase
          .from('accounts')
          .update({ current_balance: account.current_balance - transaction.amount })
          .eq('id', account.id);
      }
    } catch (error) {
      console.error('Error removing transaction:', error);
    }
  };

  const addSavingsGoal = async () => {
    if (!newGoal.name || !newGoal.target_amount) {
      alert('Please fill all required fields');
      return;
    }

    try {
      const { data, error } = await supabase
        .from('savings_goals')
        .insert([{
          name: newGoal.name,
          target_amount: parseFloat(newGoal.target_amount),
          current_amount: parseFloat(newGoal.current_amount) || 0,
          target_date: newGoal.target_date || null,
          monthly_goal: newGoal.monthly_goal ? parseFloat(newGoal.monthly_goal) : null
        }])
        .select();

      if (error) throw error;

      if (data) {
        setNewGoal({
          name: "",
          target_amount: "",
          current_amount: "0",
          target_date: "",
          monthly_goal: ""
        });
      }
    } catch (error) {
      console.error('Error adding savings goal:', error);
    }
  };

  const updateSavingsGoal = async (id: string, amount: number) => {
    try {
      const { error } = await supabase
        .from('savings_goals')
        .update({ current_amount: amount })
        .eq('id', id);

      if (error) throw error;
    } catch (error) {
      console.error('Error updating savings goal:', error);
    }
  };

  // Calculate totals
  const checkingAccount = accounts.find(a => a.type === 'checking');
  const savingsAccount = accounts.find(a => a.type === 'savings');
  
  const expenses = transactions.filter(t => t.type === 'expense');
  const income = transactions.filter(t => t.type === 'income');
  
  const totalExpenses = expenses.reduce((sum, t) => sum + Math.abs(t.amount), 0);
  const totalIncome = income.reduce((sum, t) => sum + t.amount, 0);
  const netChange = totalIncome - totalExpenses;

  const expensesByCategory = categories.map(category => ({
    category: category.name,
    amount: expenses
      .filter(expense => expense.category_id === category.id)
      .reduce((sum, expense) => sum + Math.abs(expense.amount), 0)
  })).filter(item => item.amount > 0);

  if (loading) {
    return (
      <div className="flex justify-center items-center h-screen">
        <div className="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-emerald-500"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <Card className="bg-gradient-to-r from-emerald-500 to-teal-500 text-white border-0">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <DollarSign className="h-6 w-6" />
            Budget & Finance Tracker
          </CardTitle>
        </CardHeader>
      </Card>

      {/* Balance Overview */}
      <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
        <CardHeader>
          <CardTitle>Balance Overview</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div>
              <Label>Checking Account</Label>
              <div className="text-2xl font-bold mt-1">
                ${checkingAccount?.current_balance.toFixed(2) || '0.00'}
              </div>
            </div>
            <div>
              <Label>Savings Account</Label>
              <div className="text-2xl font-bold mt-1">
                ${savingsAccount?.current_balance.toFixed(2) || '0.00'}
              </div>
            </div>
            <div className="flex flex-col justify-center items-center p-4 bg-gray-50 rounded-lg">
              <div className={`text-2xl font-bold ${netChange >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                {netChange >= 0 ? '+' : ''}${Math.abs(netChange).toFixed(2)}
              </div>
              <div className="text-sm text-gray-600">Net Change</div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Add New Transaction */}
      <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Plus className="h-5 w-5" />
            Add New Transaction
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
            <div>
              <Label>Account</Label>
              <Select 
                value={newTransaction.account_id} 
                onValueChange={(value) => 
                  setNewTransaction({...newTransaction, account_id: value})
                }
              >
                <SelectTrigger>
                  <SelectValue placeholder="Select account" />
                </SelectTrigger>
                <SelectContent>
                  {accounts.map(account => (
                    <SelectItem key={account.id} value={account.id}>
                      {account.name} ({account.type})
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div>
              <Label>Type</Label>
              <Select 
                value={newTransaction.type} 
                onValueChange={(value) => 
                  setNewTransaction({...newTransaction, type: value as 'income' | 'expense'})
                }
              >
                <SelectTrigger>
                  <SelectValue placeholder="Select type" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="income">Income</SelectItem>
                  <SelectItem value="expense">Expense</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <div>
              <Label>Amount</Label>
              <Input
                type="number"
                placeholder="0.00"
                value={newTransaction.amount}
                onChange={(e) => setNewTransaction({...newTransaction, amount: e.target.value})}
              />
            </div>
            <div>
              <Label>Date</Label>
              <Input
                type="date"
                value={newTransaction.date}
                onChange={(e) => setNewTransaction({...newTransaction, date: e.target.value})}
              />
            </div>
            <div>
              <Label>Category</Label>
              <Select 
                value={newTransaction.category_id} 
                onValueChange={(value) => 
                  setNewTransaction({...newTransaction, category_id: value})
                }
              >
                <SelectTrigger>
                  <SelectValue placeholder="Select category" />
                </SelectTrigger>
                <SelectContent>
                  {categories.map(category => (
                    <SelectItem key={category.id} value={category.id}>
                      {category.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="md:col-span-2 lg:col-span-3">
              <Label>Description</Label>
              <Input
                placeholder="What was this for?"
                value={newTransaction.description}
                onChange={(e) => setNewTransaction({...newTransaction, description: e.target.value})}
              />
            </div>
          </div>
          <Button onClick={addTransaction} className="mt-4 w-full md:w-auto">
            Add Transaction
          </Button>
        </CardContent>
      </Card>

      {/* Expense Summary */}
      <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
        <CardHeader>
          <CardTitle>Expense Summary</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="text-center mb-4">
            <div className="text-3xl font-bold text-red-600">${totalExpenses.toFixed(2)}</div>
            <div className="text-sm text-gray-600">Total Expenses This Month</div>
          </div>
          
          {expensesByCategory.length > 0 && (
            <div className="space-y-2">
              {expensesByCategory.map(item => (
                <div key={item.category} className="flex justify-between items-center">
                  <span className="text-sm">{item.category}</span>
                  <Badge variant="outline">${item.amount.toFixed(2)}</Badge>
                </div>
              ))}
            </div>
          )}
        </CardContent>
      </Card>

      {/* Recent Transactions */}
      {transactions.length > 0 && (
        <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
          <CardHeader>
            <CardTitle>Recent Transactions</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              {transactions.slice(-5).reverse().map(transaction => {
                const category = categories.find(c => c.id === transaction.category_id);
                const account = accounts.find(a => a.id === transaction.account_id);
                
                return (
                  <div key={transaction.id} className="flex justify-between items-center p-3 bg-gray-50 rounded-lg">
                    <div>
                      <div className="font-medium">{transaction.description}</div>
                      <div className="text-sm text-gray-500">
                        {category?.name || 'Uncategorized'} • {account?.name} • {format(new Date(transaction.date), 'MMM d')}
                      </div>
                    </div>
                    <div className="flex items-center gap-2">
                      <span className={`font-bold ${
                        transaction.amount < 0 ? 'text-red-600' : 'text-green-600'
                      }`}>
                        {transaction.amount < 0 ? '-' : '+'}${Math.abs(transaction.amount).toFixed(2)}
                      </span>
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => removeTransaction(transaction.id)}
                      >
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  </div>
                );
              })}
            </div>
          </CardContent>
        </Card>
      )}

      {/* Savings Goals */}
      <Card className="bg-white/70 backdrop-blur-sm border-0 shadow-lg">
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Target className="h-5 w-5" />
            Savings Goals
          </CardTitle>
        </CardHeader>
        <CardContent>
          {/* Add New Goal */}
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
            <Input
              placeholder="Goal name"
              value={newGoal.name}
              onChange={(e) => setNewGoal({...newGoal, name: e.target.value})}
            />
            <Input
              type="number"
              placeholder="Target amount"
              value={newGoal.target_amount}
              onChange={(e) => setNewGoal({...newGoal, target_amount: e.target.value})}
            />
            <Input
              type="number"
              placeholder="Current amount"
              value={newGoal.current_amount}
              onChange={(e) => setNewGoal({...newGoal, current_amount: e.target.value})}
            />
            <Input
              type="date"
              placeholder="Target date"
              value={newGoal.target_date}
              onChange={(e) => setNewGoal({...newGoal, target_date: e.target.value})}
            />
          </div>
          <Button onClick={addSavingsGoal} className="mb-6">
            Add Savings Goal
          </Button>

          {/* Goals List */}
          <div className="space-y-4">
            {savingsGoals.map(goal => (
              <div key={goal.id} className="p-4 bg-gray-50 rounded-lg">
                <div className="flex justify-between items-center mb-2">
                  <h4 className="font-medium">{goal.name}</h4>
                  <span className="text-sm text-gray-500">
                    ${goal.current_amount.toFixed(2)} / ${goal.target_amount.toFixed(2)}
                    {goal.target_date && ` • Target: ${format(new Date(goal.target_date), 'MMM yyyy')}`}
                  </span>
                </div>
                <Progress 
                  value={(goal.current_amount / goal.target_amount) * 100} 
                  className="mb-2" 
                />
                <div className="flex gap-2">
                  <Input
                    type="number"
                    placeholder="Update amount"
                    className="flex-1"
                    onBlur={(e) => {
                      const amount = parseFloat(e.target.value);
                      if (!isNaN(amount) && amount >= 0) {
                        updateSavingsGoal(goal.id, amount);
                      }
                    }}
                  />
                </div>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>
    </div>
  );
};
export default BudgetTracker;